/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.connection.validation;

import static java.util.Arrays.asList;
import static java.util.Collections.singleton;
import static org.mule.runtime.http.api.HttpConstants.Method.GET;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.http.api.HttpConstants.Method;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

/**
 * A configuration object that allows defining how the request to validate a connection must be made and what validations must
 * be done to that request's response.
 * @since 1.0
 */
public class ConnectionValidationSettings {

  private final String testConnectionPath;
  private final ExpressionLanguage expressionLanguage;

  private final Method httpMethod;
  private final Set<Integer> validStatusCodes;
  private final List<TestConnectionValidation> testConnectionValidations;
  private final MediaType responseMediaType;

  public String getTestConnectionPath() {
    return testConnectionPath;
  }

  public ExpressionLanguage getExpressionLanguage() {
    return expressionLanguage;
  }

  public Method getHttpMethod() {
    return httpMethod;
  }

  public Set<Integer> getValidStatusCodes() {
    return validStatusCodes;
  }

  public List<TestConnectionValidation> getTestConnectionValidations() {
    return testConnectionValidations;
  }

  public MediaType getResponseMediaType() {
    return responseMediaType;
  }

  /**
   * Allows creating a new instance of {@link ConnectionValidationSettings}.
   * @param testConnectionPath a path relative to the connection base uri where the validation request will be made.
   * @param expressionLanguage the expression language to be used when executing the validation expressions.
   */
  public static Builder builder(String testConnectionPath, ExpressionLanguage expressionLanguage) {
    return new Builder(testConnectionPath, expressionLanguage);
  }

  public static class Builder {

    private String testConnectionPath;
    private ExpressionLanguage expressionLanguage;

    private Method httpMethod = GET;
    private Set<Integer> validStatusCodes = singleton(200);
    private List<TestConnectionValidation> testConnectionValidations = new LinkedList<>();
    private MediaType responseMediaType;

    /**
     * Allows creating a new instance of {@link ConnectionValidationSettings}.
     * @param testConnectionPath a path relative to the connection base uri where the validation request will be made.
     * @param expressionLanguage the expression language to be used when executing the validation expressions.
     */
    public Builder(String testConnectionPath, ExpressionLanguage expressionLanguage) {
      this.testConnectionPath = testConnectionPath;
      this.expressionLanguage = expressionLanguage;
    }

    /**
     * @param httpMethod the HTTP Method to use when executing this request against the server.
     * @return this builder
     */
    public Builder httpMethod(Method httpMethod) {
      this.httpMethod = httpMethod;
      return this;
    }

    /**
     * @param statusCode all the HTTP status codes for which this request is considered a success.
     * @return this builder
     */
    public Builder validStatusCodes(Integer... statusCode) {
      this.validStatusCodes = new HashSet<>(asList(statusCode));
      return this;
    }

    /**
     * @param validationExpression    an expression to be executed on the request's response.
     *                                Must return true if the connection is valid or false if it is not.
     * @return this builder
     */
    public Builder addValidation(String validationExpression) {
      this.addValidation(validationExpression, null);
      return this;
    }

    /**
     * @param validationExpression    an expression to be executed on the request's response.
     *                                Must return true if the connection is valid or false if it is not.
     * @param errorTemplateExpression an expression to be executed when the validation is unsuccessful (When validationExpression returns false)
     *                                This expression must return an string and will be shown to the user as an error message.
     * @return this builder
     */
    public Builder addValidation(String validationExpression, String errorTemplateExpression) {
      testConnectionValidations.add(new TestConnectionValidation(validationExpression, errorTemplateExpression));
      return this;
    }

    /**
     * @param responseMediaType The expected response media type. Will be used when evaluating the validation expressions.
     *                          If set this will override the media type returned in the content-type header by the server.
     *                          This media type should be set if the server does not provide a content-type header.
     * @return this builder
     */
    public Builder responseMediaType(MediaType responseMediaType) {
      this.responseMediaType = responseMediaType;
      return this;
    }

    /**
     * @return A ConnectionValidationSettings instance as defined using this builder.
     */
    public ConnectionValidationSettings build() {
      return new ConnectionValidationSettings(this);
    }
  }

  private ConnectionValidationSettings(Builder builder) {
    this.testConnectionPath = builder.testConnectionPath;
    this.expressionLanguage = builder.expressionLanguage;

    this.httpMethod = builder.httpMethod;
    this.validStatusCodes = builder.validStatusCodes;
    this.testConnectionValidations = builder.testConnectionValidations;
    this.responseMediaType = builder.responseMediaType;
  }
}
