/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation;

import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.multipart.MultipartPayloadBuilder;
import com.mulesoft.connectivity.rest.commons.internal.multipart.DWMultipartPayloadBuilder;
import com.mulesoft.connectivity.rest.commons.internal.util.CloserCompletionCallbackDecorator;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;

import javax.inject.Inject;
import java.io.InputStream;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import static org.mule.runtime.api.metadata.MediaType.ANY;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JSON;
import static org.mule.runtime.core.api.util.IOUtils.closeQuietly;

/**
 * Base class for operations that consume a remote REST endpoint
 *
 * @since 1.0
 */
public abstract class BaseRestOperation {

  @Inject
  private ExpressionLanguage expressionLanguage;

  @Inject
  private SchedulerService schedulerService;

  /**
   * Performs the request using the given {@code builder} and {@code connection} and completes
   * the {@code callback} accordingly.
   *
   * @param config                the operation's config
   * @param connection            the connection to use
   * @param builder               the request builder
   * @param responseTimeoutMillis the request timeout in millis
   * @param streamingHelper       the {@link StreamingHelper}
   * @param callback              the operation's {@link CompletionCallback}
   */
  protected void doRequest(RestConfiguration config,
                           RestConnection connection,
                           RestRequestBuilder builder,
                           int responseTimeoutMillis,
                           StreamingHelper streamingHelper,
                           CompletionCallback<InputStream, HttpResponseAttributes> callback) {
    try {
      connection.request(builder, responseTimeoutMillis, resolveDefaultResponseMediaType(config), streamingHelper)
          .whenComplete(handleResponse(callback));
    } catch (Throwable t) {
      callback.error(t);
    }
  }

  protected MediaType resolveDefaultResponseMediaType(RestConfiguration config) {
    MediaType mediaType = getDefaultResponseMediaType();
    if (!mediaType.getCharset().isPresent()) {
      mediaType = mediaType.withCharset(config.getCharset());
    }

    return mediaType;
  }

  /**
   * Performs the request using the given {@code builder} and {@code connection} and completes
   * the {@code callback} accordingly, expecting the response to not contain a body.
   * <p>
   * If successful, the callback will be completed with an empty String as payload.
   *
   * @param connection            the connection to use
   * @param builder               the request builder
   * @param responseTimeoutMillis the request timeout in millis
   * @param streamingHelper       the {@link StreamingHelper}
   * @param callback              the operation's {@link CompletionCallback}
   */
  protected void doVoidRequest(RestConnection connection,
                               RestRequestBuilder builder,
                               int responseTimeoutMillis,
                               StreamingHelper streamingHelper,
                               CompletionCallback<String, HttpResponseAttributes> callback) {
    try {
      connection.bodylessRequest(builder, responseTimeoutMillis, ANY, streamingHelper).whenComplete(handleResponse(callback));
    } catch (Throwable t) {
      callback.error(t);
    }
  }

  private <T> BiConsumer<Result<T, HttpResponseAttributes>, Throwable> handleResponse(CompletionCallback<T, HttpResponseAttributes> callback) {
    return (r, e) -> {
      if (e != null) {
        if (schedulerService != null) {
          schedulerService.ioScheduler().submit(() -> callback.error(e));
        } else {
          callback.error(e);
        }
      } else {
        callback.success(r);
      }
    };
  }

  /**
   * Allows to make a multipart request by facilitating the construction of the multipart body through a
   * {@link MultipartPayloadBuilder}.
   * <p>
   * This operation <b>MUST NOT</b> be completed using the {@code callback} passed as a parameter, but the one
   * provided through the {@code consumer}
   *
   * @param builderConfigurer a {@link Consumer} which gives access to the {@link MultipartPayloadBuilder} and allows to configure it
   * @param callback          the operation's {@link CompletionCallback}
   * @param consumer          the {@link Consumer} which gives access to the built multipart body and the effective callback that should
   *                          complete this operation
   * @param <T>               the generic type of the {@code callback} payload type
   * @param <A>               the generic type of the {@code callback} attributes type
   */
  protected <T, A> void withMultipart(Consumer<MultipartPayloadBuilder> builderConfigurer,
                                      CompletionCallback<T, A> callback,
                                      BiConsumer<TypedValue<InputStream>, CompletionCallback<T, A>> consumer) {

    MultipartPayloadBuilder builder = new DWMultipartPayloadBuilder(expressionLanguage);
    builderConfigurer.accept(builder);

    TypedValue<InputStream> multipart = builder.build();
    try {
      consumer.accept(multipart, new CloserCompletionCallbackDecorator<>(callback, multipart.getValue()));
    } catch (Exception e) {
      closeQuietly(multipart.getValue());
      callback.error(e);
    }
  }

  /**
   * This method specifies the {@link MediaType} that should be assumed the response to have in case the remote
   * service doesn't specify a {@code Content-Type} header.
   * <p>
   * By default, this method returns {@link MediaType#APPLICATION_JSON} since it's the most common response type. However,
   * this method should be overwritten if a different type of media type is to be expected, or if you know that a certain
   * encoding will be enforced.
   *
   * @return the {@link MediaType} to assign the response in case the server doesn't specify one in its response
   */
  protected MediaType getDefaultResponseMediaType() {
    return APPLICATION_JSON;
  }

}
