/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.model.builder.common;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.getTypedValueOrNull;

import com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext;
import com.mulesoft.connectivity.rest.commons.internal.model.common.SimpleEvaluationContext;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * A fluent builder for an operation evaluation context.
 *
 * @since 1.0
 */
public class OperationEvaluationContextBuilder {

  private final static String KEY_HEADERS = "header";
  private final static String KEY_URI_PARAMETERS = "uriParameter";
  private final static String KEY_QUERY_PARAMETERS = "queryParameter";

  private final EvaluationContext globalEvaluationContext;
  private final Map<String, Object> headers = new HashMap<>();
  private final Map<String, Object> uriParameters = new HashMap<>();
  private final Map<String, Object> queryParameters = new HashMap<>();

  public OperationEvaluationContextBuilder(EvaluationContext globalEvaluationContext) {
    this.globalEvaluationContext = globalEvaluationContext;
  }

  private OperationEvaluationContextBuilder setKeyValue(String name, Object value, Map<String, Object> map) {
    map.put(name, getTypedValueOrNull(value));
    return this;
  }

  public OperationEvaluationContextBuilder header(String name, Object value) {
    return setKeyValue(name, value, headers);
  }

  public OperationEvaluationContextBuilder uriParameter(String name, Object value) {
    return setKeyValue(name, value, uriParameters);
  }

  public OperationEvaluationContextBuilder queryParameter(String name, Object value) {
    return setKeyValue(name, value, queryParameters);
  }

  private Optional<EvaluationContext> buildGlobalEvaluationContext() {
    return Optional.ofNullable(globalEvaluationContext);
  }

  public EvaluationContext build() {
    final EvaluationContext evaluationContext = new SimpleEvaluationContext(buildGlobalEvaluationContext().orElse(null));
    evaluationContext.define(KEY_HEADERS, headers);
    evaluationContext.define(KEY_URI_PARAMETERS, uriParameters);
    evaluationContext.define(KEY_QUERY_PARAMETERS, queryParameters);
    return evaluationContext;
  }
}
