/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.model.builder.dataexpressions;

import com.mulesoft.connectivity.rest.commons.internal.model.dataexpressions.DataExpression;
import com.mulesoft.connectivity.rest.commons.internal.model.dataexpressions.SequenceCompositeDataExpression;
import com.mulesoft.connectivity.rest.commons.internal.model.dataexpressions.SequenceCompositeDataExpression.Step;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * A fluent builder for a sequence data expression.
 *
 * @since 1.0
 */
public class SequenceCompositeDataExpressionBuilder implements DataExpressionBuilder<SequenceCompositeDataExpression> {

  public static class StepBuilder {

    private final String name;
    private DataExpressionBuilders dataExpressionBuilders;
    private DataExpression dataExpression;

    public StepBuilder(String name) {
      this.name = name;
    }

    public StepBuilder dataExpression(Consumer<DataExpressionBuilders> consumer) {
      dataExpressionBuilders = new DataExpressionBuilders();
      consumer.accept(dataExpressionBuilders);
      return this;
    }

    public StepBuilder withDataExpression(DataExpression dataExpression) {
      this.dataExpression = dataExpression;
      return this;
    }

    private Optional<String> buildName() {
      return Optional.ofNullable(name);
    }

    private DataExpression buildDataExpression() {
      if (dataExpressionBuilders != null && dataExpression != null) {
        throw new IllegalStateException("Both consumer and expression set");
      }
      if (dataExpressionBuilders == null && dataExpression == null) {
        throw new IllegalStateException("No consumer or expression set");
      }
      if (dataExpressionBuilders != null) {
        return dataExpressionBuilders.build();
      }
      return dataExpression;
    }

    public Step build() {
      return new Step(buildName().orElse(null), buildDataExpression());
    }
  }

  private final List<StepBuilder> stepBuilders;

  public SequenceCompositeDataExpressionBuilder() {
    stepBuilders = new ArrayList<>();
  }

  public SequenceCompositeDataExpressionBuilder step(String name, Consumer<StepBuilder> consumer) {
    final StepBuilder stepBuilder = new StepBuilder(name);
    consumer.accept(stepBuilder);
    stepBuilders.add(stepBuilder);
    return this;
  }

  public SequenceCompositeDataExpressionBuilder step(Consumer<StepBuilder> consumer) {
    return step(null, consumer);
  }

  private List<Step> buildSteps() {
    return stepBuilders.stream().map(StepBuilder::build).collect(Collectors.toList());
  }

  public SequenceCompositeDataExpression build() {
    return new SequenceCompositeDataExpression(buildSteps());
  }
}
