/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.model.dataexpressions;

import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.ATTRIBUTES_VAR;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.PAYLOAD_VAR;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.operation.Result;

import com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * A data expression that composes multiple data expressions in steps.
 * A step will always have access to the result of the previous step.
 * If the defined steps are named, further steps will also have access to its result by this name.
 *
 * @since 1.0
 */
public class SequenceCompositeDataExpression implements DataExpression {

  public static class Step {

    private final String name;
    private final DataExpression dataExpression;

    public Step(String name, DataExpression dataExpression) {
      this.name = name;
      this.dataExpression = dataExpression;
    }

    public Step(DataExpression dataExpression) {
      this(null, dataExpression);
    }

    public Optional<String> getName() {
      return Optional.ofNullable(name);
    }

    public DataExpression getDataExpression() {
      return dataExpression;
    }
  }

  private final List<Step> steps = new ArrayList<>();

  public SequenceCompositeDataExpression(List<Step> steps) {
    this.steps.addAll(steps);
  }

  @Override
  public Object evaluate(EvaluationContext evaluationContext) {
    Object result = null;

    for (Step step : steps) {
      result = step.getDataExpression().evaluate(evaluationContext);
      //TODO RSDK-262: Is unwrapping here ok?
      defineResultForNextEvaluationContext(evaluationContext, result);
      defineStepForNextEvaluationContext(evaluationContext, step, result);
    }

    return result;
  }

  //TODO RSDK-262: Could be simplified and merged with #defineResultForNextEvaluationContext
  private void defineStepForNextEvaluationContext(EvaluationContext evaluationContext, Step step, Object result) {
    if (step.getName().isPresent()) {
      final Map<String, Object> stepResult = new HashMap<>();

      if (result instanceof Result<?, ?>) {
        Result<?, ?> resultResult = (Result<?, ?>) result;
        stepResult.put(PAYLOAD_VAR, resultResult.getOutput());
        stepResult.put(ATTRIBUTES_VAR, resultResult.getAttributes());
      } else {
        stepResult.put(PAYLOAD_VAR, result);
      }

      evaluationContext.define(step.getName().get(), TypedValue.of(stepResult));
    }
  }

  private void defineResultForNextEvaluationContext(EvaluationContext evaluationContext, Object result) {
    if (result instanceof Result<?, ?>) {
      Result<?, ?> resultResult = (Result<?, ?>) result;
      evaluationContext.define(PAYLOAD_VAR, resultResult.getOutput());
      evaluationContext.define(ATTRIBUTES_VAR, resultResult.getAttributes());
    } else {
      evaluationContext.define(PAYLOAD_VAR, result);
    }
  }
}
