/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.model.valueprovider;

import static com.mulesoft.connectivity.rest.commons.internal.util.DwUtils.getCleanExpression;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isNotBlank;

import com.mulesoft.connectivity.rest.commons.internal.model.builder.dataexpressions.SequenceCompositeDataExpressionBuilder;
import com.mulesoft.connectivity.rest.commons.internal.model.common.Evaluable;
import com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext;
import com.mulesoft.connectivity.rest.commons.internal.model.dataexpressions.HttpRequestDataExpression;
import com.mulesoft.connectivity.rest.commons.internal.model.dataexpressions.SequenceCompositeDataExpression;
import com.mulesoft.connectivity.rest.commons.internal.model.resolvers.ResolverDefinition;

/**
 * Implementation of a value provider definition for the Resolver framework.
 *
 * Executes a provided http request, gets the result, and evaluates a custom script data expression
 * to generate the possible values.
 *
 * @since 1.0
 */
public class ValueProviderResolverDefinition extends ResolverDefinition implements Evaluable {

  public final static String ID_PROPERTY_KEY = "id";
  public final static String DISPLAY_NAME_PROPERTY_KEY = "name";

  private final static String EXTRACTION_EXPRESSION_PLACEHOLDER = "%{extraction-expression}";
  private final static String ID_EXPRESSION_PLACEHOLDER = "%{id-expression}";
  private final static String DISPLAY_NAME_EXPRESSION_PLACEHOLDER = "%{name-expression}";

  private final static String MAP_EXPRESSION_TEMPLATE =
      "(" + EXTRACTION_EXPRESSION_PLACEHOLDER + ") map (item, index) -> {"
          + "    " + ID_PROPERTY_KEY + ": (" + ID_EXPRESSION_PLACEHOLDER + ") as String,"
          + "    " + DISPLAY_NAME_PROPERTY_KEY + ": (" + DISPLAY_NAME_EXPRESSION_PLACEHOLDER + ") as String }";

  private final SequenceCompositeDataExpression sequence;

  public ValueProviderResolverDefinition(
                                         HttpRequestDataExpression httpRequestDataExpression,
                                         String itemExtractionExpression,
                                         String itemNameExpression,
                                         String itemValueExpression) {

    this.sequence = buildSequence(httpRequestDataExpression,
                                  itemExtractionExpression,
                                  itemNameExpression,
                                  itemValueExpression);
  }

  /**
   * Executes the provided http request and build a DW script using the provided expressions that
   * extracts the values from the server response.
   */
  @Override
  public Object evaluate(EvaluationContext evaluationContext) {
    return sequence.evaluate(evaluationContext);
  }

  private SequenceCompositeDataExpression buildSequence(HttpRequestDataExpression httpRequestDataExpression,
                                                        String itemExtractionExpression,
                                                        String itemNameExpression,
                                                        String itemValueExpression) {

    final String mapExpression = buildMapExpression(itemExtractionExpression, itemNameExpression, itemValueExpression);

    return new SequenceCompositeDataExpressionBuilder()
        .step(step -> step
            .withDataExpression(httpRequestDataExpression))
        .step(step -> step
            .dataExpression(exp -> exp
                .script(script -> script
                    .scriptExpression(dw -> dw.expression(mapExpression)))))
        .build();
  }

  private String buildMapExpression(String itemExtractionExpression, String itemNameExpression, String itemValueExpression) {
    return MAP_EXPRESSION_TEMPLATE
        //Expressions are expected to be provided between brackets - #[expression] -, so we remove those.
        .replace(EXTRACTION_EXPRESSION_PLACEHOLDER, getCleanExpression(itemExtractionExpression))
        .replace(ID_EXPRESSION_PLACEHOLDER, getCleanExpression(itemValueExpression))
        .replace(DISPLAY_NAME_EXPRESSION_PLACEHOLDER, getItemNameExpression(itemNameExpression, itemValueExpression));
  }

  private String getItemNameExpression(String itemNameExpression, String itemValueExpression) {
    //Display name expression is optional, using item value as name if it is not provided.
    String cleanNameExpression = getCleanExpression(itemNameExpression);

    if (isNotBlank(cleanNameExpression)) {
      return cleanNameExpression;
    }

    return getCleanExpression(itemValueExpression);
  }
}
