/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.datasense.sampledata;

import static com.mulesoft.connectivity.rest.commons.internal.util.ResolverUtil.createParentEvaluationContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.sdk.api.data.sample.SampleDataException;
import org.mule.sdk.api.data.sample.SampleDataProvider;
import org.mule.sdk.api.runtime.operation.Result;

import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.internal.model.builder.common.EvaluationContextBuilderFactory;
import com.mulesoft.connectivity.rest.commons.internal.model.builder.sampledata.SampleDataResolverExpressionBuilder;
import com.mulesoft.connectivity.rest.commons.internal.model.common.Evaluable;
import com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext;

import java.util.Optional;

import javax.inject.Inject;

/**
 * A base sample data implementation that uses the resolver framework internally. Executes a sample data expression and allows
 * child classes to adapt the result to its needs.
 *
 * @since 1.0
 */
abstract class BaseRestSampleDataProvider<T, U> implements SampleDataProvider<T, U> {

  @Config
  private RestConfiguration config;

  @Connection
  protected RestConnection connection;

  @Inject
  protected ExpressionLanguage expressionLanguage;

  @Override
  public String getId() {
    return getClass().getSimpleName();
  }

  /**
   * Allows building the sample data expression that will be executed by this value provider.
   * 
   * @param builder An empty builder where the value provider that is going to be executed must be built.
   */
  protected abstract void build(SampleDataResolverExpressionBuilder builder);

  // TODO: RSDK-277 getGlobalEvaluationContext, configureEvaluationContext and createEvaluationContext is repeated here and in
  // RestValueProvider.
  // We might need to extract that to a common/util/builder class.
  /**
   * Allows child classes to provide a global evaluation context. This is useful when a value provider is a global definition that
   * need to be reused by reference, in that case the returned EvaluationContext would contain the declaration of that definition.
   *
   * @return The global evaluation context for this value provider.
   */
  protected Optional<EvaluationContext> getGlobalEvaluationContext() {
    return Optional.empty();
  }

  /**
   * Allows child classes to configure the evaluation context. This is useful for defining operation specific parameters in that
   * evaluation context, i.e. parameters defined in the operation that the value provider needs to use.
   *
   * @param evaluationContextBuilderFactory The operation evaluation context builder.
   */
  protected EvaluationContext buildEvaluationContext(EvaluationContextBuilderFactory evaluationContextBuilderFactory) {
    return evaluationContextBuilderFactory.emptyContextBuilder().build();
  }

  private EvaluationContext createEvaluationContext() {
    EvaluationContext parentEvaluationContext = createParentEvaluationContext(getGlobalEvaluationContext().orElse(null),
                                                                              config, connection, expressionLanguage);

    EvaluationContextBuilderFactory evaluationContextBuilderFactory =
        new EvaluationContextBuilderFactory(parentEvaluationContext);

    return buildEvaluationContext(evaluationContextBuilderFactory);
  }

  /***
   * Transforms the output of the sample data sequence expression to a result for the corresponding sample data type.
   * 
   * @param output The sequence data expression output.
   * @return A result matching the sample data implementation output type.
   */
  protected abstract Result<T, U> transformOutputToResult(Object output);

  @Override
  public Result<T, U> getSample() throws SampleDataException {
    SampleDataResolverExpressionBuilder builder = new SampleDataResolverExpressionBuilder();
    build(builder);
    final Evaluable sampleDataResolverExpression = builder.build();
    final EvaluationContext evaluationContext = createEvaluationContext();
    final Object resultOutput = sampleDataResolverExpression.evaluate(evaluationContext);
    return transformOutputToResult(resultOutput);
  }
}
