/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging;

import static java.lang.String.format;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.DataType.NUMBER;

import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;

import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;

import java.util.List;
import java.util.function.Function;

/**
 * Implementation of {@link RestPagingProvider} for APIs which do paging based on requesting individual pages by number
 *
 * @since 1.0
 */
public class PageNumberPagingProvider extends RestPagingProvider {

  private final String pageNumberParamName;
  private final String pageCountExpression;
  private int pageNumber;

  /**
   * Creates a new instance
   *
   * @param pageNumberParamName the name of the query param that carries the page number value
   * @param initialPageNumber the initial page to get in the first request
   * @param requestFactory a {@link Function} to generate the request to be used on each page request. Each invocation should
   *        yield a different instance
   * @param expressionLanguage the app's {@link ExpressionLanguage}
   * @param streamingHelper the {@link StreamingHelper} associated to the executing operation
   * @param payloadExpression a DW expression to extract the data from the response
   * @param defaultMediaType the {@link MediaType} for the page items if the server doesn't specify one
   * @param responseTimeout the timeout for each request
   */
  public PageNumberPagingProvider(String pageNumberParamName,
                                  String pageCountExpression,
                                  int initialPageNumber,
                                  Function<RestConnection, RestRequestBuilder> requestFactory,
                                  ExpressionLanguage expressionLanguage,
                                  StreamingHelper streamingHelper,
                                  String payloadExpression,
                                  MediaType defaultMediaType,
                                  int responseTimeout) {
    super(requestFactory, expressionLanguage, streamingHelper, payloadExpression, defaultMediaType, responseTimeout);
    this.pageNumberParamName = pageNumberParamName;
    this.pageCountExpression = pageCountExpression;
    pageNumber = initialPageNumber;
  }

  @Override
  protected void configureRequest(RestRequestBuilder requestBuilder) {
    requestBuilder.addQueryParam(pageNumberParamName, String.valueOf(pageNumber));
  }

  @Override
  protected void onPage(List<TypedValue<String>> page, TypedValue<String> rawPage,
                        HttpResponseAttributes httpResponseAttributes) {
    if (pageCountExpression != null && pageNumber == extractPageCount(rawPage, httpResponseAttributes)) {
      stopPaging();
    } else {
      pageNumber++;
    }
  }

  private Integer extractPageCount(TypedValue<String> rawPage, HttpResponseAttributes httpResponseAttributes) {
    Integer pageCount = (Integer) evaluate(rawPage, pageCountExpression, NUMBER, httpResponseAttributes).getValue();
    if (pageCount == null) {
      throw new MuleRuntimeException(createStaticMessage(
                                                         format("Failed to extract pageCount from expression: %s",
                                                                pageCountExpression)));
    }
    return pageCount;
  }
}
