/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.source;

import static java.util.stream.Collectors.toList;

import java.util.LinkedList;
import java.util.List;

/**
 * This class allows defining the parameter bindings of a trigger or value provider. A binding defines the value or expression
 * that will be used to generate the value of a parameter. Parameters and values defined here will be sent to the server in the
 * poll request.
 *
 * @since 1.0
 */
public class RequestParameterBinding {

  private final List<Binding> headers = new LinkedList<>();
  private final List<Binding> queryParams = new LinkedList<>();
  private final List<Binding> uriParams = new LinkedList<>();

  public List<Binding> getHeaders() {
    return headers;
  }

  public List<Binding> getQueryParams() {
    return queryParams;
  }

  public List<Binding> getUriParams() {
    return uriParams;
  }

  public List<String> getAllBindingExpressions() {
    List<String> allBindings = headers.stream().map(Binding::getValue).collect(toList());
    allBindings.addAll(queryParams.stream().map(Binding::getValue).collect(toList()));
    allBindings.addAll(uriParams.stream().map(Binding::getValue).collect(toList()));

    return allBindings;
  }

  public void addHeaderBinding(String parameterName, String expression) {
    headers.add(new Binding(parameterName, expression));
  }

  public void addQueryParamBinding(String parameterName, String expression) {
    queryParams.add(new Binding(parameterName, expression));
  }

  public void addUriParamBinding(String parameterName, String expression) {
    uriParams.add(new Binding(parameterName, expression));
  }

  public static class Binding {

    private final String key;
    private final String value;

    public Binding(String key, String value) {
      this.key = key;
      this.value = value;
    }

    public String getKey() {
      return key;
    }

    public String getValue() {
      return value;
    }
  }
}
