/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.datasense.valueprovider;

import static com.mulesoft.connectivity.rest.commons.internal.model.valueprovider.ValueProviderResolverDefinition.DISPLAY_NAME_PROPERTY_KEY;
import static com.mulesoft.connectivity.rest.commons.internal.model.valueprovider.ValueProviderResolverDefinition.ID_PROPERTY_KEY;
import static com.mulesoft.connectivity.rest.commons.internal.util.ResolverUtil.createParentEvaluationContext;
import static org.mule.runtime.extension.api.values.ValueBuilder.newValue;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.value.Value;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.values.ValueProvider;
import org.mule.runtime.extension.api.values.ValueResolvingException;

import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.internal.model.builder.common.EvaluationContextBuilderFactory;
import com.mulesoft.connectivity.rest.commons.internal.model.builder.valueprovider.ValueProviderResolverExpressionBuilder;
import com.mulesoft.connectivity.rest.commons.internal.model.common.Evaluable;
import com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext;

import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

import javax.inject.Inject;

import org.json.JSONArray;
import org.json.JSONObject;

/**
 * A value provider implementation that uses the resolver framework internally. Executes a value provider expression adapts it to
 * the Java SDK API.
 *
 * @since 1.0
 */
public abstract class RestValueProvider implements ValueProvider {

  @Config
  private RestConfiguration config;

  @Connection
  protected RestConnection connection;

  @Inject
  private ExpressionLanguage expressionLanguage;

  /**
   * Allows building the value provider expression that will be executed by this value provider.
   * 
   * @param builder An empty builder where the value provider that is going to be executed must be built.
   */
  protected abstract void build(ValueProviderResolverExpressionBuilder builder);

  // TODO: RSDK-277 getGlobalEvaluationContext, configureEvaluationContext and createEvaluationContext is repeated here and in
  // RestValueProvider.
  // We might need to extract that to a common/util/builder class.
  /**
   * Allows child classes to provide a global evaluation context. This is useful when a value provider is a global definition that
   * need to be reused by reference, in that case the returned EvaluationContext would contain the declaration of that definition.
   *
   * @return The global evaluation context for this value provider.
   */
  protected Optional<EvaluationContext> getGlobalEvaluationContext() {
    return Optional.empty();
  }

  /**
   * Allows child classes to configure the evaluation context. This is useful for defining operation specific parameters in that
   * evaluation context, i.e. parameters defined in the operation that the value provider needs to use.
   *
   * @param evaluationContextBuilderFactory The evaluation context builder factory.
   */
  protected EvaluationContext buildEvaluationContext(EvaluationContextBuilderFactory evaluationContextBuilderFactory) {
    return evaluationContextBuilderFactory.emptyContextBuilder().build();
  }

  @Override
  public Set<Value> resolve() throws ValueResolvingException {
    ValueProviderResolverExpressionBuilder builder = new ValueProviderResolverExpressionBuilder();
    build(builder);

    final Evaluable valueProviderResolverExpression = builder.build();

    final Object resultOutput = valueProviderResolverExpression.evaluate(createEvaluationContext());
    return getValuesFromResult(resultOutput);
  }

  private EvaluationContext createEvaluationContext() {
    EvaluationContext parentEvaluationContext = createParentEvaluationContext(getGlobalEvaluationContext().orElse(null),
                                                                              config, connection, expressionLanguage);

    EvaluationContextBuilderFactory evaluationContextBuilderFactory =
        new EvaluationContextBuilderFactory(parentEvaluationContext);

    return buildEvaluationContext(evaluationContextBuilderFactory);
  }

  private Set<Value> getValuesFromResult(Object resultOutput) {
    Set<Value> values = new HashSet<>();

    for (Object o : getJsonArray(resultOutput)) {
      values.add(getValueFromJsonObject((JSONObject) o));
    }

    return values;
  }

  private Value getValueFromJsonObject(JSONObject jsonObject) {
    return newValue(jsonObject.getString(ID_PROPERTY_KEY))
        .withDisplayName(jsonObject.getString(DISPLAY_NAME_PROPERTY_KEY))
        .build();
  }

  private JSONArray getJsonArray(Object resultOutput) {
    final String s = ((TypedValue<?>) resultOutput).getValue().toString();
    return new JSONArray(s);
  }
}
