/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.datasense.metadata.common;

import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.PAYLOAD_VAR;
import static com.mulesoft.connectivity.rest.commons.internal.util.RequestStreamingUtils.doRequestAndConsumeString;
import static com.mulesoft.connectivity.rest.commons.internal.util.ResolverUtil.createParentEvaluationContext;
import static org.mule.metadata.api.model.MetadataFormat.JSON;
import static org.mule.runtime.api.metadata.DataType.STRING;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JSON;
import static org.mule.runtime.api.metadata.resolving.FailureCode.INVALID_CONFIGURATION;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MetadataContext;
import org.mule.runtime.api.metadata.MetadataResolvingException;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.metadata.resolving.FailureCode;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.http.api.HttpConstants;

import com.mulesoft.connectivity.rest.commons.api.binding.HttpRequestBinding;
import com.mulesoft.connectivity.rest.commons.api.binding.ParameterBinding;
import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.dw.DWBindings;
import com.mulesoft.connectivity.rest.commons.api.dw.HttpResponseDWBinding;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.internal.model.builder.common.EvaluationContextBuilderFactory;
import com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;

public abstract class JsonCustomFieldsHttpMetadataResolver {

  protected abstract String getSchemaPath();

  protected abstract String getMethod();

  protected abstract String getPath();

  protected abstract String getTransformationScript();

  protected abstract String getSelector();

  public MetadataType resolveMetadataType(MetadataContext metadataContext, Object key)
      throws MetadataResolvingException {
    try {
      RestConnection connection = getConnection(metadataContext);
      RestConfiguration configuration = getConfiguration(metadataContext);
      RestRequestBuilder requestBuilder = getRestRequestBuilder(key, connection, configuration);

      String customFieldSchema = doRequestAndTransformSchema(connection, configuration, requestBuilder);
      MetadataType metadataType = TypeParameterResolverUtil.resolveType(configuration.getExpressionLanguageMetadataService(),
                                                                        customFieldSchema, getSelector(), getSchemaPath(), JSON);
      return metadataType;
    } catch (Exception e) {
      throw new MetadataResolvingException("Error generating metadata output type", FailureCode.UNKNOWN, e);
    }
  }

  private String doRequestAndTransformSchema(RestConnection connection, RestConfiguration configuration,
                                             RestRequestBuilder requestBuilder) {
    Result<TypedValue<String>, HttpResponseAttributes> response =
        doRequestAndConsumeString(connection, configuration, requestBuilder, APPLICATION_JSON,
                                  configuration.getExpressionLanguage());
    HttpResponseDWBinding httpResponseDWBinding =
        new HttpResponseDWBinding(response.getOutput(), response.getAttributes().orElse(null));

    return configuration.getExpressionLanguage().evaluate(getTransformationScript(), DataType.JSON_STRING,
                                                          BindingContext.builder()
                                                              .addBinding(PAYLOAD_VAR, response.getOutput())
                                                              .addBinding(DWBindings.RESPONSE.getBinding(),
                                                                          TypedValue.of(httpResponseDWBinding))
                                                              .build())
        .getValue().toString();
  }

  private RestRequestBuilder getRestRequestBuilder(Object key, RestConnection connection, RestConfiguration configuration) {
    RestRequestBuilder restRequestBuilder =
        new RestRequestBuilder(connection.getBaseUri(), getPath(), HttpConstants.Method.valueOf(getMethod()));
    if (configuration.getResponseInterceptorDescriptor() != null) {
      restRequestBuilder.responseInterceptorDescriptor(configuration.getResponseInterceptorDescriptor());
    }

    EvaluationContext parentEvaluationContext =
        createParentEvaluationContext(null, configuration, connection, configuration.getExpressionLanguage());
    EvaluationContextBuilderFactory evaluationContextBuilderFactory =
        new EvaluationContextBuilderFactory(parentEvaluationContext);
    EvaluationContext evaluationContext = buildEvaluationContext(evaluationContextBuilderFactory, key);
    HttpRequestBinding requestBindings = getRequestBindings();

    for (ParameterBinding binding : requestBindings.getUriParams()) {
      restRequestBuilder.addUriParam(binding.getKey(),
                                     ((TypedValue<String>) evaluationContext.resolve(binding.getValue(), STRING)).getValue());
    }

    for (ParameterBinding binding : requestBindings.getQueryParams()) {
      restRequestBuilder.addQueryParam(binding.getKey(),
                                       ((TypedValue<String>) evaluationContext.resolve(binding.getValue(), STRING)).getValue());
    }

    for (ParameterBinding binding : requestBindings.getHeaders()) {
      restRequestBuilder.addHeader(binding.getKey(),
                                   ((TypedValue<String>) evaluationContext.resolve(binding.getValue(), STRING)).getValue());
    }

    return restRequestBuilder;
  }

  protected HttpRequestBinding getRequestBindings() {
    return new HttpRequestBinding();
  }

  protected EvaluationContext buildEvaluationContext(EvaluationContextBuilderFactory evaluationContextBuilderFactory,
                                                     Object key) {
    return evaluationContextBuilderFactory.emptyContextBuilder().build();
  }

  private RestConnection getConnection(MetadataContext context) throws ConnectionException {
    return context.<RestConnection>getConnection().orElseThrow(
                                                               () -> new ConnectionException("No connection supplied for metadata generation"));
  }

  protected RestConfiguration getConfiguration(MetadataContext context) throws MetadataResolvingException {
    return context.<RestConfiguration>getConfig().orElseThrow(
                                                              () -> new MetadataResolvingException("No configuration supplied for metadata generation",
                                                                                                   INVALID_CONFIGURATION));
  }

}
