/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.source;


import com.mulesoft.connectivity.rest.commons.api.backoff.RandomizedExponentialBackoffCaller;
import com.mulesoft.connectivity.rest.commons.api.backoff.RetriableCallerConfiguration;
import com.mulesoft.connectivity.rest.commons.api.binding.HttpRequestBinding;
import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.dw.DWBindings;
import com.mulesoft.connectivity.rest.commons.api.dw.HttpResponseDWBinding;
import com.mulesoft.connectivity.rest.commons.api.error.RequestException;
import com.mulesoft.connectivity.rest.commons.api.error.SourceStartingException;
import com.mulesoft.connectivity.rest.commons.api.operation.ConfigurationOverrides;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.api.operation.paging.RestPagingProvider;
import com.mulesoft.connectivity.rest.commons.internal.util.DwUtils;

import static com.mulesoft.connectivity.rest.commons.api.error.RestError.TOO_MANY_REQUESTS;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.CONFIG;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.CONNECTION;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.CONNECTOR_OVERRIDES;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.CONTEXT_KEY_PARAMETERS;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.closeStream;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.getTypedValueOrNull;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isBlank;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.commons.internal.util.StreamUtils.resolveCursorProvider;
import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static org.mule.runtime.api.el.BindingContext.builder;
import static org.mule.runtime.api.metadata.DataType.STRING;

import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.el.ValidationResult;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.connectivity.oauth.AccessTokenExpiredException;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.source.PollContext;
import org.mule.runtime.extension.api.runtime.source.PollingSource;
import org.mule.runtime.extension.api.runtime.source.SourceCallbackContext;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Supplier;

import javax.inject.Inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Base class for defining a polling source that consumes a remote REST endpoint in a connector.
 *
 * @since 1.0
 */
public abstract class PaginatedBasedRestPollingSource
    extends PollingSource<InputStream, Object> {

  private static final Logger LOGGER = LoggerFactory.getLogger(PaginatedBasedRestPollingSource.class);
  private static final String ITEM_BINDING = "item";

  @Config
  protected RestConfiguration config;

  @Connection
  private ConnectionProvider<RestConnection> connectionProvider;

  @Inject
  private ExpressionLanguage expressionLanguage;

  protected RestConnection connection;

  protected final String watermarkExpression;
  protected final String identityExpression;
  protected final String requestBodyExpression;
  protected final String eventExpression;
  protected final String startValueExpression;

  private TypedValue<?> startValue;
  private Map<String, Object> parameterBindings;

  protected RestPagingProvider baseOperationPagingProvider;

  @Parameter
  @Optional // default empty, no max set, infinite
  private int restSdk_maxRetries;

  @Parameter
  @Optional(defaultValue = "900000") // 15 minutes
  private int restSdk_maxWaitingValue;

  @ParameterGroup(name = CONNECTOR_OVERRIDES)
  protected ConfigurationOverrides overrides;

  private RandomizedExponentialBackoffCaller<RestPagingProvider.PaginationResponse> randomizedExponentialBackoffCaller;

  public PaginatedBasedRestPollingSource(String watermarkExpression, String identityExpression,
                                         String requestBodyExpression, String eventExpression, String startValueExpression) {

    this.watermarkExpression = watermarkExpression;
    this.identityExpression = identityExpression;
    this.requestBodyExpression = requestBodyExpression;
    this.eventExpression = eventExpression;
    this.startValueExpression = startValueExpression;
  }

  @Override
  protected void doStart() throws MuleException {
    validateExpression(watermarkExpression);
    validateExpression(identityExpression);
    validateExpression(requestBodyExpression);
    validateExpression(startValueExpression);

    for (String bindingExpression : getParameterBinding().getAllBindingExpressions()) {
      if (DwUtils.isExpression(bindingExpression)) {
        validateExpression(bindingExpression);
      }
    }

    connection = connectionProvider.connect();
    evaluateStartValue();

    parameterBindings = (Map<String, Object>) resolveCursorProvider((Map) getParameterValues());

    randomizedExponentialBackoffCaller = new RandomizedExponentialBackoffCaller<RestPagingProvider.PaginationResponse>(
                                                                                                                       new RetriableCallerConfiguration(restSdk_maxRetries,
                                                                                                                                                        restSdk_maxWaitingValue)) {

      @Override
      public boolean isRetriable(com.mulesoft.connectivity.rest.commons.api.backoff.Result result) {
        if (result.isError() && result.getResult().getClass().equals(RequestException.class) &&
            ((RequestException) result.getResult()).getType().equals(TOO_MANY_REQUESTS)) {
          return true;
        }
        return false;
      }
    };

    doInstancesStart();
  }

  /**
   * Start method to be overridden on each instance, at least making the baseOperationPagingProvider initialization.
   */
  protected abstract void doInstancesStart();

  private void evaluateStartValue() {
    BindingContext.Builder builder = builder();
    addParametersBinding(builder);
    addConfigAndConnectionBinding(builder);
    BindingContext bindingContext = builder.build();
    startValue = getStartValue(bindingContext);
  }

  /**
   * Returns the initial value start value will hold. Defaults to the expression parameterized as the startValueExpression, but it
   * can execute any code (such as a http call to obtain data).
   *
   * @param bindingContext the context to access the parameters, config or connection in the DW engine
   * @return a TypedValue with the information that will be available later on in the `startValue` binding in the trigger later on
   */
  protected TypedValue<?> getStartValue(BindingContext bindingContext) {
    return getExpressionLanguage().evaluate(startValueExpression, bindingContext);
  }

  @Override
  protected void doStop() {
    connectionProvider.disconnect(connection);
  }

  /**
   * Returns the parameter binding configuration of this RestPollingSource.
   */
  protected abstract HttpRequestBinding getParameterBinding();

  /**
   * Returns a MultiMap containing all the parameters this source exposes to the user. Each entry must contain the parameter name
   * as key and its value represented as a TypedValue.
   */
  protected abstract Map<String, Object> getParameterValues();

  /**
   * Returns the request path for this RestPollingSource with placeholders for its uri parameters. i.e: /user/{username}/events
   */
  protected abstract String getPathTemplate();

  protected String getId() {
    return getClass().getSimpleName();
  }

  /**
   * @return The injected config
   */
  protected RestConfiguration getConfig() {
    return config;
  }

  /**
   * @return The injected expression language
   */
  protected ExpressionLanguage getExpressionLanguage() {
    return expressionLanguage;
  }

  /**
   * @return The injected connection
   */
  protected RestConnection getConnection() {
    return connection;
  }


  private RestPagingProvider.PaginationResponse getItemsByPageBackoff(PollContext<InputStream, Object> pollContext) {
    try {
      return randomizedExponentialBackoffCaller.call(() -> getBackoffResultGetCaller(pollContext));
    } catch (InterruptedException e) {
      LOGGER.warn(format("Trigger '%s': Mule runtime exception found while executing poll: '%s'", getId(), e.getMessage()), e);
      baseOperationPagingProvider.cleanPagingAttributes();
      return new RestPagingProvider.PaginationResponse(null, null, emptyList());
    }
  }

  private com.mulesoft.connectivity.rest.commons.api.backoff.Result getBackoffResultGetCaller(PollContext<InputStream, Object> pollContext) {
    try {
      return new com.mulesoft.connectivity.rest.commons.api.backoff.Result<>(baseOperationPagingProvider
          .getFullPage(connection));
    } catch (AccessTokenExpiredException e) {
      baseOperationPagingProvider.cleanPagingAttributes();
      LOGGER
          .info(format("Trigger '%s': about to notify access token expiration to runtime...",
                       getId()),
                e);
      pollContext.onConnectionException(new ConnectionException(e));
      LOGGER
          .info(format("Trigger '%s': access token expiration notified to runtime.",
                       getId()),
                e);
      return new com.mulesoft.connectivity.rest.commons.api.backoff.Result<>(e);
    } catch (RequestException e) {
      return new com.mulesoft.connectivity.rest.commons.api.backoff.Result<>(e);
    } catch (MuleRuntimeException e) {
      baseOperationPagingProvider.cleanPagingAttributes();
      LOGGER.warn(
                  format("Trigger '%s': Mule runtime exception found while executing poll: '%s'",
                         getId(), e.getMessage()),
                  e);
      return new com.mulesoft.connectivity.rest.commons.api.backoff.Result<>(e);
    }
  }

  @Override
  public void poll(PollContext<InputStream, Object> pollContext) {
    final Serializable watermark = pollContext.getWatermark().orElse(null);

    RestPagingProvider.PaginationResponse paginationResponse = getItemsByPageBackoff(pollContext);
    while (!paginationResponse.getItems().isEmpty()) {
      for (TypedValue<String> item : paginationResponse.getItems()) {
        pollContext
            .accept(getPollItemConsumer(paginationResponse.getFullPage(), paginationResponse.getAttributes(), watermark, item));
      }
      paginationResponse = getItemsByPageBackoff(pollContext);
    }
    baseOperationPagingProvider.cleanPagingAttributes();
  }

  private Consumer<PollContext.PollItem<InputStream, Object>> getPollItemConsumer(
                                                                                  TypedValue<String> fullPage,
                                                                                  HttpResponseAttributes attributes,
                                                                                  Serializable watermark,
                                                                                  TypedValue<String> item) {
    return pollItem -> {
      TypedValue<InputStream> inputStreamTypedValue = getEvent(item);

      Result<InputStream, Object> itemResult =
          Result.<InputStream, Object>builder()
              .output(inputStreamTypedValue.getValue())
              .mediaType(inputStreamTypedValue.getDataType().getMediaType())
              .build();

      pollItem.setResult(itemResult);

      if (isNotBlank(watermarkExpression)) {
        pollItem.setWatermark(getItemWatermark(fullPage, attributes, watermark, item));
      }

      if (isNotBlank(identityExpression)) {
        pollItem.setId(getIdentity(fullPage, attributes, watermark, item));
      }
    };
  }

  protected DataType getWatermarkDataType() {
    return STRING;
  }

  private Serializable getItemWatermark(TypedValue<String> fullPage, HttpResponseAttributes attributes,
                                        Serializable currentWatermark, TypedValue<String> item) {
    return (Serializable) expressionLanguage
        .evaluate(watermarkExpression, getWatermarkDataType(), buildContext(fullPage, attributes, currentWatermark, item))
        .getValue();
  }

  private String getIdentity(TypedValue<String> fullPage, HttpResponseAttributes attributes,
                             Serializable currentWatermark, TypedValue<String> item) {
    return (String) expressionLanguage
        .evaluate(identityExpression, STRING, buildContext(fullPage, attributes, currentWatermark, item))
        .getValue();
  }

  private TypedValue<InputStream> getEvent(TypedValue<String> item) {

    if (eventExpression != null) {
      TypedValue<?> eventTransform = expressionLanguage.evaluate(eventExpression, item.getDataType(),
                                                                 builder().addBinding(ITEM_BINDING, item).build());

      item = new TypedValue<>(eventTransform.getValue().toString(), eventTransform.getDataType());
    }

    return new TypedValue<>(new ByteArrayInputStream(item.getValue().getBytes()), item.getDataType());
  }

  private void validateExpression(String expression) throws SourceStartingException {
    if (isBlank(expression)) {
      return;
    }

    ValidationResult validationResult = expressionLanguage.validate(expression);

    if (!validationResult.isSuccess()) {
      throw new SourceStartingException(format("Expression is not valid: %s", expression));
    }
  }

  private BindingContext buildContext(
                                      TypedValue<String> fullPage,
                                      HttpResponseAttributes attributes,
                                      Serializable currentWatermark,
                                      TypedValue<String> item) {
    HttpResponseDWBinding httpResponseDWBinding =
        new HttpResponseDWBinding(fullPage, attributes);
    BindingContext.Builder builder =
        builder()
            .addBinding("payload", fullPage)
            .addBinding(DWBindings.RESPONSE.getBinding(), TypedValue.of(httpResponseDWBinding))
            .addBinding("watermark", TypedValue.of(currentWatermark))
            .addBinding("startValue", startValue);
    addParametersBinding(builder);
    addConfigAndConnectionBinding(builder);

    if (item != null) {
      builder.addBinding(ITEM_BINDING, item);
    }

    return builder.build();
  }

  private void addParametersBinding(BindingContext.Builder builder) {
    builder
        .addBinding(CONTEXT_KEY_PARAMETERS, TypedValue.of(parameterBindings));
  }

  private void addConfigAndConnectionBinding(BindingContext.Builder builder) {
    builder
        .addBinding(CONFIG, getTypedValueOrNull(config.getBindings()))
        .addBinding(CONNECTION, getTypedValueOrNull(connection.getBindings()));
  }

  @Override
  public void onRejectedItem(Result<InputStream, Object> result,
                             SourceCallbackContext callbackContext) {
    if (result.getOutput() != null) {
      closeStream(result.getOutput());
    }
    LOGGER.debug("Item Rejected");
  }

  protected MediaType getDefaultResponseMediaType() {
    return MediaType.APPLICATION_JSON;
  }
}
