/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.interception;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.MediaType;

import com.mulesoft.connectivity.rest.commons.api.interception.HttpResponseInterceptor;
import com.mulesoft.connectivity.rest.commons.api.interception.descriptor.CompositeHttpResponseInterceptorDescriptor;
import com.mulesoft.connectivity.rest.commons.api.interception.descriptor.ExpressionHttpResponseInterceptorDescriptor;
import com.mulesoft.connectivity.rest.commons.api.interception.descriptor.HttpResponseInterceptorDescriptor;
import com.mulesoft.connectivity.rest.commons.api.interception.descriptor.HttpResponseInterceptorDescriptorVisitor;
import com.mulesoft.connectivity.rest.commons.api.interception.descriptor.PreconditionHttpResponseInterceptorDelegateDescriptor;
import com.mulesoft.connectivity.rest.commons.api.streaming.StreamingHelper;
import com.mulesoft.connectivity.rest.commons.internal.interception.expression.ExpressionHttpResponseInterceptor;
import com.mulesoft.connectivity.rest.commons.internal.interception.expression.PreconditionHttpResponseInterceptorDelegate;

import java.util.List;
import java.util.stream.Collectors;

public class HttpResponseInterceptorFactory {

  public HttpResponseInterceptor newHttpResponseInterceptor(HttpResponseInterceptorDescriptor descriptor,
                                                            MediaType defaultResponseMediaType,
                                                            StreamingHelper streamingHelper,
                                                            ExpressionLanguage expressionLanguage) {
    HttpResponseInterceptorFactoryVisitor httpResponseInterceptorFactoryVisitor =
        new HttpResponseInterceptorFactoryVisitor(defaultResponseMediaType, streamingHelper, expressionLanguage);
    descriptor.accept(httpResponseInterceptorFactoryVisitor);
    return httpResponseInterceptorFactoryVisitor.getHttpResponseInterceptor();
  }

  private class HttpResponseInterceptorFactoryVisitor implements HttpResponseInterceptorDescriptorVisitor {

    private HttpResponseInterceptor httpResponseInterceptor;

    private MediaType defaultResponseMediaType;
    private StreamingHelper streamingHelper;
    private ExpressionLanguage expressionLanguage;

    public HttpResponseInterceptorFactoryVisitor(MediaType defaultResponseMediaType, StreamingHelper streamingHelper,
                                                 ExpressionLanguage expressionLanguage) {
      this.defaultResponseMediaType = requireNonNull(defaultResponseMediaType, "defaultResponseMediaType cannot be null");
      this.streamingHelper = streamingHelper;
      this.expressionLanguage = requireNonNull(expressionLanguage, "expressionLanguage cannot be null");
    }

    @Override
    public void visit(ExpressionHttpResponseInterceptorDescriptor descriptor) {
      ExpressionHttpResponseInterceptor.ExpressionHttpResponseInterceptorBuilder builder =
          ExpressionHttpResponseInterceptor.builder()
              .matchExpression(descriptor.getMatchExpression())
              .defaultResponseMediaType(defaultResponseMediaType);

      ofNullable(descriptor.getStatusCodeExpression()).ifPresent(expression -> builder.statusCodeExpression(expression));
      ofNullable(descriptor.getReasonPhraseExpression()).ifPresent(expression -> builder.reasonPhraseExpression(expression));
      ofNullable(descriptor.getBodyExpression()).ifPresent(expression -> builder.bodyExpression(expression));
      ofNullable(descriptor.getHeadersExpression()).ifPresent(expression -> builder.headersExpression(expression));

      ofNullable(streamingHelper).ifPresent(expression -> builder.streamingHelper(streamingHelper));
      builder.expressionLanguage(expressionLanguage);

      httpResponseInterceptor = builder.build();
    }

    @Override
    public void visit(PreconditionHttpResponseInterceptorDelegateDescriptor descriptor) {
      HttpResponseInterceptorFactoryVisitor delegateVisitor =
          new HttpResponseInterceptorFactoryVisitor(defaultResponseMediaType, streamingHelper, expressionLanguage);
      descriptor.getDelegateDescriptor().accept(delegateVisitor);
      httpResponseInterceptor = new PreconditionHttpResponseInterceptorDelegate(statusCode -> descriptor
          .getPrecondition()
          .match(statusCode),
                                                                                delegateVisitor.getHttpResponseInterceptor());
    }

    @Override
    public void visit(CompositeHttpResponseInterceptorDescriptor descriptor) {
      List<HttpResponseInterceptor> httpResponseInterceptors = descriptor.getHttpResponseInterceptorDescriptors()
          .stream()
          .map(itemDescriptor -> {
            HttpResponseInterceptorFactoryVisitor delegateVisitor =
                new HttpResponseInterceptorFactoryVisitor(defaultResponseMediaType, streamingHelper, expressionLanguage);
            itemDescriptor.accept(delegateVisitor);
            return delegateVisitor.getHttpResponseInterceptor();
          })
          .collect(Collectors.toList());
      this.httpResponseInterceptor = new CompositeHttpResponseInterceptor(httpResponseInterceptors);
    }

    public HttpResponseInterceptor getHttpResponseInterceptor() {
      return httpResponseInterceptor;
    }
  }
}
