/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.interception.expression;

import static com.mulesoft.connectivity.rest.commons.api.dw.DWBindings.BODY;
import static com.mulesoft.connectivity.rest.commons.api.dw.DWBindings.HEADERS;
import static com.mulesoft.connectivity.rest.commons.api.dw.DWBindings.REASON_PHRASE;
import static com.mulesoft.connectivity.rest.commons.api.dw.DWBindings.REQUEST;
import static com.mulesoft.connectivity.rest.commons.api.dw.DWBindings.STATUS_CODE;
import static com.mulesoft.connectivity.rest.commons.api.dw.DWBindings.MEDIATYPE;
import static com.mulesoft.connectivity.rest.commons.internal.interception.model.RepeatableHttpResponse.newRepeatableHttpResponse;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.resolveCharset;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.el.ExpressionLanguageSession;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.interception.HttpRequest;
import com.mulesoft.connectivity.rest.commons.api.interception.HttpResponseInterceptor;
import com.mulesoft.connectivity.rest.commons.api.streaming.StreamingHelper;
import com.mulesoft.connectivity.rest.commons.internal.interception.model.HttpEntityCursorStreamProviderBased;
import com.mulesoft.connectivity.rest.commons.internal.interception.model.RepeatableHttpResponse;
import com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils;

/**
 * A base implementation for a {@link HttpResponseInterceptor} that relies on DW expressions. It allows use of DW expressions to
 * define the matching condition and transformations for statusCode, reasonPhrase, headers and body.
 * <p/>
 * {@link BindingContext} provided on each expression has the original {@link HttpResponse} data with the following variables:
 * <ul>
 * <li>statusCode: int</li>
 * <li>reasonPhrase: String</li>
 * <li>headers: MultiMap<String, String></li>
 * <li>body: InputStream</li>
 * </ul>
 *
 * @since 1.0
 */
public abstract class BaseExpressionHttpResponseInterceptor implements HttpResponseInterceptor {

  private MediaType defaultResponseMediaType;
  private ExpressionLanguage expressionLanguage;
  protected StreamingHelper streamingHelper;

  protected BaseExpressionHttpResponseInterceptor(MediaType defaultResponseMediaType, ExpressionLanguage expressionLanguage,
                                                  StreamingHelper streamingHelper) {
    requireNonNull(defaultResponseMediaType);
    requireNonNull(expressionLanguage);
    requireNonNull(streamingHelper);

    this.defaultResponseMediaType = defaultResponseMediaType;
    this.expressionLanguage = expressionLanguage;
    this.streamingHelper = streamingHelper;
  }

  /**
   * Creates a {@link BindingContext} with data resolved from the {@link HttpResponseInterceptor} and {@link HttpResponse} as
   * variables:
   * <ul>
   * <li>request: HttpResponseInterceptor</li>
   * <li>statusCode: int</li>
   * <li>reasonPhrase: String</li>
   * <li>headers: MultiMap<String, String></li>
   * <li>body: Any</li>
   * </ul>
   *
   * @param httpRequest the request to be injected to the binding context.
   * @param httpResponse the response to retrieve the values for variables related to response.
   *
   * @return a {@link BindingContext}.
   */
  private BindingContext createBindingContext(HttpRequest httpRequest, RepeatableHttpResponse httpResponse) {
    BindingContext.Builder bindingContextBuilder = BindingContext.builder();

    bindingContextBuilder.addBinding(REQUEST.getBinding(), TypedValue.of(httpRequest));

    MediaType mediaType = getMediaType(httpResponse);
    bindingContextBuilder.addBinding(STATUS_CODE.getBinding(), TypedValue.of(httpResponse.getStatusCode()));
    bindingContextBuilder.addBinding(REASON_PHRASE.getBinding(), TypedValue.of(httpResponse.getReasonPhrase()));
    bindingContextBuilder.addBinding(HEADERS.getBinding(), TypedValue.of(httpResponse.getHeaders()));
    bindingContextBuilder
        .addBinding(BODY.getBinding(), new TypedValue(((HttpEntityCursorStreamProviderBased) httpResponse.getEntity())
            .getCursorStreamProvider(), DataType.builder()
                .mediaType(mediaType)
                .charset(resolveCharset(empty(), mediaType))
                .build()));
    bindingContextBuilder.addBinding(MEDIATYPE.getBinding(), TypedValue.of(mediaType.toRfcString()));

    return bindingContextBuilder.build();
  }

  protected MediaType getMediaType(HttpResponse httpResponse) {
    return RestSdkUtils.getMediaType(httpResponse, defaultResponseMediaType);
  }

  /**
   * Evaluates an expression giving the bindingContext and dataType.
   *
   * @param expression to be evaluated.
   * @param session expression language session.
   * @param dataType the output {@link DataType} for the result of the expression evaluation.
   * @param <T> type of the output {@link TypedValue<T>}.
   * @return the result of the evaluation as {@link TypedValue<T>}.
   */
  protected <T> TypedValue<T> evaluate(String expression, ExpressionLanguageSession session, DataType dataType) {
    return doEvaluate(expression, session, dataType);
  }

  private <T> TypedValue<T> doEvaluate(String expression, ExpressionLanguageSession session, DataType dataType) {
    return (TypedValue<T>) session.evaluate(expression, dataType);
  }

  @Override
  public HttpResponse intercept(HttpRequest httpRequest, HttpResponse httpResponse) {
    RepeatableHttpResponse repeatableHttpResponse = newRepeatableHttpResponse(httpResponse, streamingHelper);
    BindingContext bindingContext = createBindingContext(httpRequest, repeatableHttpResponse);
    try (ExpressionLanguageSession session = expressionLanguage.openSession(bindingContext)) {
      return doIntercept(httpRequest, repeatableHttpResponse, session, bindingContext, expressionLanguage);
    }
  }

  protected abstract RepeatableHttpResponse doIntercept(HttpRequest httpRequest, RepeatableHttpResponse repeatableHttpResponse,
                                                        ExpressionLanguageSession session,
                                                        BindingContext bindingContext,
                                                        ExpressionLanguage expressionLanguage);
}
