/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.interception.model;

import static java.util.Objects.requireNonNull;

import org.mule.api.annotation.NoExtend;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.streaming.StreamingHelper;

import java.util.Collection;

/**
 * Implementation for {@link HttpResponse} to allow making repeatable its {@link HttpEntity} content by using
 * {@link com.mulesoft.connectivity.rest.commons.api.streaming.StreamingHelper} to make the content repeatable.
 */
@NoExtend
public class RepeatableHttpResponse implements HttpResponse {

  private HttpResponse response;

  RepeatableHttpResponse(HttpResponse response) {
    requireNonNull(response, "response cannot be null.");
    this.response = response;
  }

  public static RepeatableHttpResponse newRepeatableHttpResponse(HttpResponse response, StreamingHelper streamingHelper) {
    if (response instanceof RepeatableHttpResponse) {
      return (RepeatableHttpResponse) response;
    }
    return new RepeatableHttpResponse(HttpResponse.builder()
        .statusCode(response.getStatusCode())
        .reasonPhrase(response.getReasonPhrase())
        .headers(response.getHeaders())
        .entity(toRepeatableStreamingHttpEntity(response, streamingHelper))
        .build());
  }

  /**
   * Creates an {@link HttpEntity} that is backed up with a {@link CursorStreamProvider} to allow streaming the entity content.
   */
  private static HttpEntity toRepeatableStreamingHttpEntity(HttpResponse response,
                                                            StreamingHelper streamingHelper) {
    HttpEntity entity = response.getEntity();
    // If the passed entity is already a cursorStream based entity provider, and it is not closed we return this.
    if (entity instanceof HttpEntityCursorStreamProviderBased &&
        !((HttpEntityCursorStreamProviderBased) entity).getCursorStreamProvider().isClosed()) {
      return entity;
    }

    CursorStreamProvider payload = streamingHelper.resolveCursorStreamProvider(entity.getContent());
    return new HttpEntityCursorStreamProviderBased(payload, entity.getBytesLength());
  }

  public void close() {
    ((HttpEntityCursorStreamProviderBased) getEntity()).close();
  }

  @Override
  public int getStatusCode() {
    return response.getStatusCode();
  }

  @Override
  public String getReasonPhrase() {
    return response.getReasonPhrase();
  }

  @Override
  public HttpEntity getEntity() {
    return response.getEntity();
  }

  @Override
  public Collection<String> getHeaderNames() {
    return response.getHeaderNames();
  }

  @Override
  public boolean containsHeader(String headerName) {
    return response.containsHeader(headerName);
  }

  @Override
  public String getHeaderValue(String headerName) {
    return response.getHeaderValue(headerName);
  }

  @Override
  @Deprecated
  public String getHeaderValueIgnoreCase(String headerName) {
    return response.getHeaderValueIgnoreCase(headerName);
  }

  @Override
  public Collection<String> getHeaderValues(String headerName) {
    return response.getHeaderValues(headerName);
  }

  @Override
  @Deprecated
  public Collection<String> getHeaderValuesIgnoreCase(String headerName) {
    return response.getHeaderValuesIgnoreCase(headerName);
  }

  @Override
  public MultiMap<String, String> getHeaders() {
    return response.getHeaders();
  }

  public HttpEntityCursorStreamProviderBased getRepeatableEntity() {
    return (HttpEntityCursorStreamProviderBased) getEntity();
  }

}
