/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging;

import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;

import java.util.List;
import java.util.Map;
import java.util.function.Function;

/**
 * Implementation of {@link RestPagingProvider} for APIs which do paging based on an offset query param.
 *
 * @since 1.0
 */
public class OffsetPagingProvider extends RestPagingProvider {

  private final String offsetParamName;
  private int offset;
  private final int initialOffset;

  /**
   * Creates a new instance
   *
   * @param offsetParamName the name of the query param that carries the offset value
   * @param initialOffset the initial offset to apply on the first request
   * @param requestFactory a {@link Function} to generate the request to be used on each page request. Each invocation should
   *        yield a different instance
   * @param expressionLanguage the app's {@link ExpressionLanguage}
   * @param payloadExpression a DW expression to extract the data from the response
   * @param defaultMediaType the {@link MediaType} for the page items if the server doesn't specify one
   * @param responseTimeout the timeout for each request
   */
  public OffsetPagingProvider(String offsetParamName,
                              int initialOffset,
                              Function<RestConnection, RestRequestBuilder> requestFactory,
                              ExpressionLanguage expressionLanguage,
                              String payloadExpression,
                              MediaType defaultMediaType,
                              Map<String, Object> parameterBindings,
                              Map<String, Object> customParameterBindings,
                              int responseTimeout) {
    super(requestFactory, expressionLanguage, payloadExpression,
          defaultMediaType, parameterBindings, customParameterBindings, responseTimeout);
    this.offsetParamName = offsetParamName;
    offset = initialOffset;
    this.initialOffset = initialOffset;
  }

  public void cleanPagingAttributes() {
    super.cleanPagingAttributes();
    offset = initialOffset;
  }

  @Override
  protected void configureRequest(RestRequestBuilder requestBuilder) {
    requestBuilder.addQueryParam(offsetParamName, offset);
  }

  @Override
  protected void onPage(List<TypedValue<String>> page, TypedValue<String> rawPage,
                        HttpResponseAttributes httpResponseAttributes) {
    offset += page.size();
  }
}
