/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.util;

import static com.mulesoft.connectivity.rest.commons.internal.metadatamodel.handler.SchemaHandler.CUSTOM_FIELD_LOCATION;

import org.mule.metadata.api.annotation.MetadataFormatPropertiesAnnotation;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.IntersectionType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.UnionType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;

import java.util.Optional;

public class MetadataUtils {

  public enum LocationType {
    ITEM, ARRAY
  }

  public static class ObjectHolder<T> {

    private T obj;

    public ObjectHolder(T obj) {
      this.obj = obj;
    }

    public T get() {
      return obj;
    }

    public void set(T obj) {
      this.obj = obj;
    }
  }


  public static Optional<MetadataType> findObjectLocation(MetadataType metadataType) {
    ObjectHolder<MetadataType> objectHolder = new ObjectHolder<>(null);
    metadataType.accept(new MetadataTypeVisitor() {

      @Override
      protected void defaultVisit(MetadataType metadataType) {
        processLocation(metadataType);
      }

      private void processLocation(MetadataType metadataType) {
        final String location = getCustomFieldLocation(metadataType);
        if (location != null) {
          if (overrideLocation(location)) {
            throw new IllegalStateException(String.format("Attempt to override location [%s] with [%s].", getLocation().get(),
                                                          location));
          } else {
            setLocation(metadataType);
          }
        }
      }

      private boolean overrideLocation(String location) {
        return getLocation().isPresent() && !location.equals(getCustomFieldLocation(getLocation().get()));
      }

      private Optional<MetadataType> getLocation() {
        return Optional.ofNullable(objectHolder.get());
      }

      private void setLocation(MetadataType metadataType) {
        objectHolder.set(metadataType);
      }

      @Override
      public void visitIntersection(IntersectionType intersectionType) {
        super.visitIntersection(intersectionType);
        if (!getLocation().isPresent()) {
          intersectionType.getTypes().forEach(metadataType1 -> metadataType1.accept(this));
        } ;
      }

      @Override
      public void visitUnion(UnionType unionType) {
        super.visitUnion(unionType);
        if (!getLocation().isPresent()) {
          unionType.getTypes().forEach(metadataType1 -> metadataType1.accept(this));
        } ;
      }

      @Override
      public void visitArrayType(ArrayType arrayType) {
        super.visitArrayType(arrayType);
        arrayType.getType().accept(this);
      }
    });
    return Optional.ofNullable(objectHolder.get());
  }

  private static String getCustomFieldLocation(MetadataType metadataType) {
    return metadataType.getAnnotation(MetadataFormatPropertiesAnnotation.class)
        .map(MetadataFormatPropertiesAnnotation::getValue)
        .map(properties -> properties.get(CUSTOM_FIELD_LOCATION))
        .orElse(null);
  }
}
