/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.model.common;

import com.mulesoft.connectivity.rest.commons.internal.model.resolvers.ResolverDeclaration;

import java.util.Map;
import java.util.Optional;

/**
 * A context where an evaluable item can be evaluated. The evaluable item using this evaluation context will get the properties
 * and variables it needs to generate a result from here.
 *
 * @since 1.0
 */
@SuppressWarnings("rawtypes")
public interface EvaluationContext {

  String PROPERTY_CONFIGURATION = "PROPERTY_CONFIGURATION";
  String PROPERTY_CONNECTION = "PROPERTY_CONNECTION";
  String PROPERTY_EXPRESSION_LANGUAGE = "PROPERTY_EXPRESSION_LANGUAGE";

  /**
   * Evaluates an expression in this context and get it's result.
   * 
   * @param expression The expression to evaluate.
   * @return The expression result aafter evaluating it in this context.
   */
  Object resolve(String expression);

  /**
   * Get a the value of a property declared in this context by name. If the property is not declared in this context a null value
   * should be returned.
   * 
   * @param property The name of the property in this context.
   * @return The value of the property.
   */
  Object resolveProperty(String property);

  /**
   * Allows defining a property in this context.
   * 
   * @param property The name of the property to define.
   * @param value The value of the property.
   */
  void defineProperty(String property, Object value);

  /**
   * Allows defining a variable in this context.
   * 
   * @param variable The name of the variable to define.
   * @param value The value of the variable.
   */
  void define(String variable, Object value);

  /**
   * Gets a resolver declaration from this evaluation context.
   * 
   * @param declarationName The declared name of the resolver.
   * @return A resolver declaration corresponding to the specified name.
   */
  ResolverDeclaration resolveDeclaration(String declarationName);

  /**
   * Allows defining a resolver declaration in this context.
   * 
   * @param resolverDeclaration The resolver declaration to define in this context.
   */
  void declare(ResolverDeclaration resolverDeclaration);

  /**
   * Gets all the variables defined in this context.
   * 
   * @return All the variables defined in this context.
   */
  Map<String, Object> getVariables();

  /**
   * Get the global evaluation context for this context.
   * 
   * @return The global evaluation context.
   */
  Optional<EvaluationContext> getGlobalEvaluationContext();
}
