/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.metadata;

import static com.mulesoft.connectivity.rest.commons.internal.metadata.RestSchemaHelper.generateSchema;

import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.utils.MetadataTypeUtils;

import com.mulesoft.connectivity.rest.commons.internal.metadata.handler.ParsingContext;
import com.mulesoft.connectivity.rest.commons.internal.metadata.handler.RestHandlerManager;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.Optional;

import org.everit.json.schema.Schema;
import org.json.JSONObject;
import org.json.JSONTokener;

/***
 * This class loads the metadata structure from a JSONSchema as JsonTypeLoader would do but allows defining a MetadataFormat for
 * it instead of setting it to JSON.
 */
public class RestJsonTypeLoader implements TypeLoader {

  private JsonSchemaLoader jsonSchemaLoader;
  private MetadataFormat metadataFormat;

  public RestJsonTypeLoader(File schemaFile, MetadataFormat metadataFormat) {
    jsonSchemaLoader = new JsonSchemaFileLoader(schemaFile);
    this.metadataFormat = metadataFormat;
  }

  public RestJsonTypeLoader(String schemaData, MetadataFormat metadataFormat) {
    this(schemaData, null, metadataFormat);
  }

  public RestJsonTypeLoader(String schemaData, URI baseURI, MetadataFormat metadataFormat) {
    jsonSchemaLoader = new JsonSchemaStringLoader(schemaData, baseURI);
    this.metadataFormat = metadataFormat;
  }

  private JsonSchemaLoader getJsonSchemaLoader() {
    return jsonSchemaLoader;
  }

  @Override
  public Optional<MetadataType> load(String identifier) {
    return load(identifier, null);
  }

  @Override
  public Optional<MetadataType> load(String identifier, String typeAlias) {
    final Schema jsonSchema = getJsonSchemaLoader().loadSchema();
    final TypeBuilder<?> typeBuilder = createRestHandlerManager().handle(jsonSchema, new ParsingContext());
    MetadataTypeUtils.addTypeAlias(typeBuilder, typeAlias);
    return Optional.of(typeBuilder.build());
  }

  protected RestHandlerManager createRestHandlerManager() {
    return new RestHandlerManager(metadataFormat);
  }

  private interface JsonSchemaLoader {

    Schema loadSchema();
  }

  private static class JsonSchemaFileLoader implements JsonSchemaLoader {

    private File schemaFile;

    public JsonSchemaFileLoader(File schemaFile) {
      this.schemaFile = schemaFile;
    }

    @Override
    public Schema loadSchema() {
      try {
        try (InputStream inputStream = new FileInputStream(schemaFile)) {
          final JSONObject rawSchema = new JSONObject(new JSONTokener(inputStream));
          return generateSchema(rawSchema, schemaFile.toURI());
        }
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    }
  }

  private static class JsonSchemaStringLoader implements JsonSchemaLoader {

    private final URI baseURI;
    private String schemaData;

    public JsonSchemaStringLoader(String schemaData, URI baseURI) {
      this.schemaData = schemaData;
      this.baseURI = baseURI;
    }

    @Override
    public Schema loadSchema() {
      final JSONObject rawSchema = new JSONObject(new JSONTokener(schemaData));
      return generateSchema(rawSchema, baseURI);
    }
  }
}
