/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.metadata.handler;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isNotBlank;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.everit.json.schema.ObjectSchema;
import org.everit.json.schema.Schema;

public class ObjectHandler implements SchemaHandler {

  @Override
  public boolean handles(Schema schema) {
    return schema instanceof ObjectSchema;
  }

  @Override
  public TypeBuilder<?> handle(Schema schema, BaseTypeBuilder root, RestHandlerManager restHandlerManager,
                               ParsingContext parsingContext) {

    final ObjectSchema objectSchema = (ObjectSchema) schema;
    final ObjectTypeBuilder objectMetadataBuilder = root.objectType();
    parsingContext.registerBuilder(objectSchema, objectMetadataBuilder);

    final String description = objectSchema.getDescription();
    if (isNotEmpty(description)) {
      objectMetadataBuilder.description(description);
    }

    final String title = objectSchema.getTitle();
    if (isNotEmpty(title)) {
      objectMetadataBuilder.label(title);
    }

    final String id = objectSchema.getId();
    if (isNotEmpty(id)) {
      objectMetadataBuilder.id(id);
    }

    final Set<Map.Entry<String, Schema>> properties = objectSchema.getPropertySchemas().entrySet();
    final List<String> requiredProperties = objectSchema.getRequiredProperties();
    // Add named properties
    for (Map.Entry<String, Schema> property : properties) {
      final ObjectFieldTypeBuilder field = objectMetadataBuilder.addField().key(property.getKey());
      field.required(requiredProperties.contains(property.getKey()));
      final Schema value = property.getValue();
      field.value(restHandlerManager.handle(value, parsingContext));

      final String fieldDescription = value.getDescription();
      if (isNotEmpty(fieldDescription)) {
        field.description(fieldDescription);
      }

      final String fieldTitle = value.getTitle();
      if (isNotEmpty(fieldTitle)) {
        field.label(fieldTitle);
      }
    }
    final Map<Pattern, Schema> patternProperties = objectSchema.getPatternProperties();
    // Sort them by pattern
    final Collection<Map.Entry<Pattern, Schema>> entries =
        patternProperties.entrySet().stream().sorted((o1, o2) -> o1.getKey().toString().compareTo(o2.getKey().toString()))
            .collect(Collectors.toList());
    for (Map.Entry<Pattern, Schema> patternSchemaEntry : entries) {
      final ObjectFieldTypeBuilder field = objectMetadataBuilder.addField();
      field.key(patternSchemaEntry.getKey());
      final Schema value = patternSchemaEntry.getValue();
      field.value(restHandlerManager.handle(value, parsingContext));
    }
    if (objectSchema.permitsAdditionalProperties()) {
      objectMetadataBuilder.openWith(restHandlerManager.handle(objectSchema.getSchemaOfAdditionalProperties(), parsingContext));
    }
    return objectMetadataBuilder;
  }

  private boolean isNotEmpty(String description) {
    return description != null && !description.isEmpty();
  }
}
