/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isBlank;
import static java.lang.String.format;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.DataType.STRING;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;

import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.internal.RestConstants;
import com.mulesoft.connectivity.rest.commons.internal.util.LinkHeaderUtils;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;

import java.util.List;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Implementation of {@link RestPagingProvider} for APIs which do paging based on a next URL provided in the response of each
 * request.
 * <p>
 * Request number 1 will obtain the first page plus the next URL embedded in the response. Said URL will be the request number 2,
 * which will reply with the next page and a new URL which overwrites the prior one. The process continues until no new URL is
 * provided or the next page comes back empty.
 *
 * @since 1.0
 */
public class HypermediaPagingProvider extends RestPagingProvider {

  private static final Logger LOGGER = LoggerFactory.getLogger(HypermediaPagingProvider.class);

  private final String nextUrlExpression;

  private boolean firstPage = true;
  private String nextUrl;

  /**
   * Creates a new instance
   *
   * @param nextUrlExpression the DW expression used to extract the next URL from the response
   * @param requestFactory a {@link Function} to generate the request to be used on each page request. Each invocation should
   *        yield a different instance
   * @param expressionLanguage the app's {@link ExpressionLanguage}
   * @param streamingHelper the {@link StreamingHelper} associated to the executing operation
   * @param payloadExpression a DW expression to extract the data from the response
   * @param defaultMediaType the {@link MediaType} for the page items if the server doesn't specify one
   * @param responseTimeout the timeout for each request
   */
  public HypermediaPagingProvider(String nextUrlExpression,
                                  Function<RestConnection, RestRequestBuilder> requestFactory,
                                  ExpressionLanguage expressionLanguage,
                                  StreamingHelper streamingHelper,
                                  String payloadExpression,
                                  MediaType defaultMediaType,
                                  int responseTimeout) {
    super(requestFactory, expressionLanguage, streamingHelper, payloadExpression, defaultMediaType, responseTimeout);
    this.nextUrlExpression = nextUrlExpression;
  }

  @Override
  protected void configureRequest(RestRequestBuilder requestBuilder) {
    if (!firstPage) {
      if (isBlank(nextUrl)) {
        stopPaging();
        return;
      }

      requestBuilder.setUri(nextUrl);
    }
  }

  @Override
  protected void onPage(List<TypedValue<String>> page, TypedValue<String> rawPage, HttpResponseAttributes responseAttributes) {
    firstPage = false;

    if (isBlank(rawPage.getValue())) {
      nextUrl = null;
      stopPaging();
    } else {
      nextUrl = extractNexUrl(rawPage, responseAttributes);
    }

  }

  private String extractNexUrl(TypedValue<String> rawPage, HttpResponseAttributes responseAttributes) {
    try {
      return (String) evaluate(rawPage, nextUrlExpression, STRING, responseAttributes).getValue();
    } catch (Exception e) {
      throw new MuleRuntimeException(createStaticMessage(
                                                         format("Failed to extract nextUrl from expression: %s",
                                                                nextUrlExpression)));
    }
  }

  @Override
  protected BindingContext toBindingContext(TypedValue<?> content, HttpResponseAttributes httpResponseAttributes) {
    BindingContext.Builder builder = BindingContext.builder()
        .addBinding(RestConstants.PAYLOAD_VAR, content);

    if (httpResponseAttributes != null) {
      builder.addBinding(RestConstants.ATTRIBUTES_VAR, TypedValue.of(httpResponseAttributes));
      try {
        String linkHeader = httpResponseAttributes.getHeaders().get(RestConstants.LINK);
        builder.addBinding(RestConstants.LINK, TypedValue.of(LinkHeaderUtils.parseLinkHeader(linkHeader)));
      } catch (Exception e) {
        LOGGER.warn("Failed to parse Link header", e);
      }
    }

    return builder.build();
  }

}
