/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging;

import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.ATTRIBUTES_VAR;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.PAYLOAD_VAR;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.consumeStringAndClose;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isBlank;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.resolveCharset;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.toList;
import static com.mulesoft.connectivity.rest.commons.internal.util.SplitPayloadUtils.split;
import static java.lang.String.format;
import static java.lang.Thread.currentThread;
import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.expression.ExpressionRuntimeException;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;

import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.api.binding.HttpResponseBinding;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;

import java.io.InputStream;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;

/**
 * Provides the base functionality for paginating the results of a REST endpoint.
 * <p>
 * Different APIs use different paging strategies which are tackled in implementations of this class. This class provides the
 * basics as:
 *
 * <ul>
 * <li>Executing one request per page to be obtained</li>
 * <li>Ability to parse the response to extract the data and useful metadata as the total results count (if provided)</li>
 * <li>Handling streaming</li>
 * <li>Handling the nuances of the {@link PagingProvider} contract</li>
 * <li>Handling encodings and media types</li>
 * </ul>
 * <p>
 * Each item in the obtained page will be returned as an individual {@link TypedValue}, with the proper mime type and encoding
 * set.
 *
 * @since 1.0
 */
public abstract class RestPagingProvider implements PagingProvider<RestConnection, TypedValue<String>> {

  protected final ExpressionLanguage expressionLanguage;
  protected final int responseTimeout;
  protected final MediaType defaultMediaType;

  private final Function<RestConnection, RestRequestBuilder> requestFactory;
  private final StreamingHelper streamingHelper;
  private final String payloadExpression;

  private boolean stopPaging = false;

  private HttpResponseBinding responseBinding = null;

  /**
   * Creates a new instance
   *
   * @param requestFactory a {@link Function} to generate the request to be used on each page request. Each invocation should
   *        yield a different instance
   * @param expressionLanguage the app's {@link ExpressionLanguage}
   * @param streamingHelper the {@link StreamingHelper} associated to the executing operation
   * @param payloadExpression a DW expression to extract the data from the response
   * @param defaultMediaType the {@link MediaType} for the page items if the server doesn't specify one
   * @param responseTimeout the timeout for each request
   */
  public RestPagingProvider(Function<RestConnection, RestRequestBuilder> requestFactory,
                            ExpressionLanguage expressionLanguage,
                            StreamingHelper streamingHelper,
                            String payloadExpression,
                            MediaType defaultMediaType,
                            int responseTimeout) {

    this.requestFactory = requestFactory;
    this.streamingHelper = streamingHelper;
    this.expressionLanguage = expressionLanguage;
    this.responseTimeout = responseTimeout;
    this.payloadExpression = payloadExpression;
    this.defaultMediaType = defaultMediaType;
  }

  /**
   * Allows specifying a transformation to be applied to the server response before extracting the items and paging them.
   * 
   * @return This paging provider
   */
  public RestPagingProvider withResponseBinding(HttpResponseBinding responseBinding) {
    this.responseBinding = responseBinding;
    return this;
  }

  /**
   * Depending on the paging strategy, different parameters or headers may need to be passed. This method receives the raw output
   * of the {@link #requestFactory} in order to configure the custom parameters that apply/
   *
   * @param requestBuilder a {@link RestRequestBuilder}
   */
  protected abstract void configureRequest(RestRequestBuilder requestBuilder);

  @Override
  public final List<TypedValue<String>> getPage(RestConnection connection) {
    if (stopPaging) {
      return emptyList();
    }

    final RestRequestBuilder requestBuilder = requestFactory.apply(connection);
    configureRequest(requestBuilder);

    Result<InputStream, HttpResponseAttributes> result;
    try {
      result = connection.request(requestBuilder, responseTimeout, defaultMediaType, streamingHelper).get();
    } catch (ExecutionException e) {
      Throwable cause = e.getCause();
      if (cause instanceof RuntimeException) {
        throw (RuntimeException) cause;
      }

      throw new MuleRuntimeException(createStaticMessage(cause.getMessage()), cause);
    } catch (InterruptedException e) {
      currentThread().interrupt();
      throw new MuleRuntimeException(e);
    }

    TypedValue<String> rawPage = consumeStringAndClose(result.getOutput(),
                                                       result.getMediaType().orElse(defaultMediaType),
                                                       resolveCharset(result.getMediaType(), defaultMediaType));

    TypedValue<String> postProcessedPage;
    if (isBlank(rawPage.getValue()) || stopPaging) {
      return emptyList();
    } else {
      postProcessedPage = postProcess(rawPage);
    }

    List<TypedValue<String>> page = extractPayload(postProcessedPage);

    onPage(page, postProcessedPage, result.getAttributes().orElse(null));

    return page;
  }

  private TypedValue<String> postProcess(TypedValue<String> rawPage) {
    if (responseBinding == null || responseBinding.getBody() == null || isBlank(responseBinding.getBody().getValue())) {
      return rawPage;
    }

    String transformationExpression = responseBinding.getBody().getValue();

    TypedValue<?> payload;
    try {
      payload = expressionLanguage.evaluate(transformationExpression, rawPage.getDataType(), toBindingContext(rawPage));
    } catch (ExpressionRuntimeException e) {
      throw new MuleRuntimeException(createStaticMessage(
                                                         format("Failed to transform server result using the expression '%s'",
                                                                transformationExpression)));
    }

    return consumeStringAndClose(
                                 payload.getValue(),
                                 rawPage.getDataType().getMediaType(),
                                 resolveCharset(Optional.of(rawPage.getDataType().getMediaType()), defaultMediaType));
  }

  @Override
  public Optional<Integer> getTotalResults(RestConnection connection) {
    return empty();
  }

  @Override
  public final void close(RestConnection connection) throws MuleException {
    doClose(connection);
  }

  /**
   * Invoked each time a page is obtained.
   *
   * @param page the obtained page
   */
  protected abstract void onPage(List<TypedValue<String>> page, TypedValue<String> rawPage,
                                 HttpResponseAttributes responseAttributes);

  protected void stopPaging() {
    stopPaging = true;
  }

  /**
   * Invoked when {@link #close(RestConnection)} is invoked
   *
   * @param connection the current connection
   * @throws MuleException
   */
  protected void doClose(RestConnection connection) throws MuleException {
    // empty default implementation
  }

  protected <T> TypedValue<T> evaluate(TypedValue<String> content, String expression, DataType expectedOutputType,
                                       HttpResponseAttributes httpResponseAttributes) {
    return (TypedValue<T>) expressionLanguage.evaluate(expression, expectedOutputType,
                                                       toBindingContext(content, httpResponseAttributes));
  }

  private List<TypedValue<String>> extractPayload(TypedValue<String> content) {
    TypedValue<?> payload;
    try {
      payload = expressionLanguage.evaluate(payloadExpression, content.getDataType(), toBindingContext(content));
    } catch (ExpressionRuntimeException e) {
      throw new MuleRuntimeException(createStaticMessage(
                                                         format("Failed to extract payload from expression: %s",
                                                                payloadExpression)));
    }
    Iterator<TypedValue<?>> iterator = split(expressionLanguage, payload, payloadExpression);
    return toList(iterator, content.getDataType().getMediaType(), defaultMediaType);
  }

  private BindingContext toBindingContext(TypedValue<?> content) {
    return toBindingContext(content, null);
  }

  protected BindingContext toBindingContext(TypedValue<?> content, HttpResponseAttributes httpResponseAttributes) {
    BindingContext.Builder builder = BindingContext.builder()
        .addBinding(PAYLOAD_VAR, content);

    if (httpResponseAttributes != null) {
      builder.addBinding(ATTRIBUTES_VAR, TypedValue.of(httpResponseAttributes));
    }

    return builder.build();
  }
}
