/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.util;

import static java.lang.Character.isWhitespace;
import static java.nio.charset.Charset.defaultCharset;
import static java.util.Optional.of;
import static org.mule.metadata.xml.api.SchemaCollector.getInstance;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JAVA;
import static org.mule.runtime.core.api.functional.Either.left;
import static org.mule.runtime.core.api.functional.Either.right;
import static org.mule.runtime.core.api.util.IOUtils.closeQuietly;
import static org.mule.runtime.core.api.util.IOUtils.toByteArray;

import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.impl.BaseMetadataType;
import org.mule.metadata.json.api.JsonTypeLoader;
import org.mule.metadata.xml.api.SchemaCollector;
import org.mule.metadata.xml.api.XmlTypeLoader;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.CursorProvider;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.core.api.functional.Either;
import org.mule.runtime.core.api.util.IOUtils;
import org.mule.runtime.extension.api.runtime.operation.Result;

import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

/**
 * Utility methods
 *
 * @since 1.0
 */
public final class RestSdkUtils {

  private RestSdkUtils() {}

  /**
   * Closes the given {@code cursorProvider} and releases all associated resources
   *
   * @param cursorProvider a cursor provider
   */
  public static void closeAndRelease(CursorProvider<?> cursorProvider) {
    cursorProvider.close();
    cursorProvider.releaseResources();
  }

  /***
   * Closes the given {@code stream} which can be either a {@link CursorProvider} or an {@link InputStream}.
   *
   * Null values or instances of other classes will be ignored.
   *
   * @param stream a stream
   */
  public static void closeStream(Object stream) {
    if (stream instanceof CursorProvider) {
      closeAndRelease((CursorProvider<?>) stream);
    } else if (stream instanceof InputStream) {
      closeQuietly((InputStream) stream);
    }
  }

  /**
   * Consumes the contents of the given {@code stream} as a String and closes it.
   * <p>
   * The {@code stream} can be either a {@link CursorStreamProvider} or an {@link InputStream}. {@link IllegalArgumentException}
   * will be thrown if the {@code stream} is an instance of any other class.
   * <p>
   * Notice that in the case of {@link CursorStreamProvider}, this method will close the provider meaning that it will no longer
   * be able to yield new cursors.
   *
   * @param stream a {@link CursorStreamProvider} or an {@link InputStream}
   * @param targetCharset the encoding to use when reading the String and that will be set to the TypedValue
   * @param targetMediaType the media type that will be set to the TypedValue
   * @return the contents as a String
   * @throws IllegalArgumentException if {@code stream} is not of the expected types.
   */
  public static TypedValue<String> consumeStringAndClose(Object stream, MediaType targetMediaType, Charset targetCharset) {
    return consumeStringTransformAndClose(stream, targetCharset, targetMediaType, targetCharset);
  }

  /**
   * Consumes the contents of the given {@code stream} as a {@link ByteArrayInputStream} and closes it.
   * <p>
   * The {@code stream} can be either a {@link CursorStreamProvider} or an {@link InputStream}. {@link IllegalArgumentException}
   * will be thrown if the {@code stream} is an instance of any other class.
   * <p>
   * Notice that in the case of {@link CursorStreamProvider}, this method will close the provider meaning that it will no longer
   * be able to yield new cursors.
   *
   * @param stream a {@link CursorStreamProvider} or an {@link InputStream}
   * @param targetCharset the encoding to use when reading the String and that will be set to the TypedValue
   * @param targetMediaType the media type that will be set to the TypedValue
   * @return the contents as an offline stream
   * @throws IllegalArgumentException if {@code stream} is not of the expected types.
   */
  public static TypedValue<InputStream> consumeToOfflineStreamAndClose(Object stream, MediaType targetMediaType,
                                                                       Charset targetCharset) {
    TypedValue<String> string = consumeStringAndClose(stream, targetMediaType, targetCharset);
    return new TypedValue<>(new ByteArrayInputStream(string.getValue().getBytes()),
                            DataType.builder()
                                .type(InputStream.class)
                                .mediaType(string.getDataType().getMediaType())
                                .build(),
                            string.getByteLength());
  }

  /**
   * Consumes the contents of the given {@code stream} as a String and closes it.
   * <p>
   * The {@code stream} can be either a {@link CursorStreamProvider} or an {@link InputStream}. {@link IllegalArgumentException}
   * will be thrown if the {@code stream} is an instance of any other class.
   * <p>
   * Notice that in the case of {@link CursorStreamProvider}, this method will close the provider meaning that it will no longer
   * be able to yield new cursors.
   *
   * @param stream a {@link CursorStreamProvider} or an {@link InputStream}
   * @param sourceCharset the encoding to use when reading the String
   * @param targetCharset the encoding that will be set to the TypedValue
   * @param targetMediaType the media type that will be set to the TypedValue
   * @return the contents as a String
   * @throws IllegalArgumentException if {@code stream} is not of the expected types.
   */
  public static TypedValue<String> consumeStringTransformAndClose(Object stream, Charset sourceCharset, MediaType targetMediaType,
                                                                  Charset targetCharset) {
    if (stream == null) {
      return toTypedValue("", targetMediaType, targetCharset);
    }

    if (stream instanceof String) {
      return toTypedValue((String) stream, targetMediaType, targetCharset);
    }

    Either<CursorStreamProvider, InputStream> content;

    if (stream instanceof CursorStreamProvider) {
      content = left((CursorStreamProvider) stream);
    } else if (stream instanceof InputStream) {
      content = right((InputStream) stream);
    } else {
      throw new IllegalArgumentException("Cannot consume stream of unsupported type: " + stream.getClass().getName());
    }

    return content.reduce(provider -> {
      try {
        return doConsumeAndClose(provider.openCursor(), sourceCharset, targetMediaType, targetCharset);
      } finally {
        closeAndRelease(provider);
      }
    }, in -> doConsumeAndClose(in, sourceCharset, targetMediaType, targetCharset));
  }

  private static TypedValue<String> doConsumeAndClose(InputStream stream, Charset sourceCharset, MediaType targetMediaType,
                                                      Charset targetCharset) {
    try {
      return toTypedValue(IOUtils.toString(stream, sourceCharset), targetMediaType, targetCharset);
    } finally {
      closeQuietly(stream);
    }
  }

  public static TypedValue<String> toTypedValue(String value, MediaType mediaType, Charset encoding) {
    return new TypedValue<>(value, DataType.builder().mediaType(mediaType).charset(encoding).build());
  }

  public static <T> TypedValue<T> toPayloadTypedValue(Result<T, ?> result) {
    return new TypedValue<>(result.getOutput(), DataType.builder()
        .type(result.getOutput().getClass())
        .mediaType(result.getMediaType().orElse(APPLICATION_JAVA))
        .charset(resolveCharset(result.getMediaType(), APPLICATION_JAVA))
        .build());
  }

  public static byte[] outputToByteArray(org.mule.sdk.api.runtime.operation.Result<?, HttpResponseAttributes> result) {
    byte[] outputByteArray;
    if (result.getOutput() instanceof String) {
      outputByteArray = ((String) result.getOutput()).getBytes();
    } else if (result.getOutput() instanceof InputStream) {
      InputStream inputStream = (InputStream) result.getOutput();
      try {
        outputByteArray = toByteArray(inputStream);
      } finally {
        closeQuietly(inputStream);
      }
    } else {
      throw new MuleRuntimeException(createStaticMessage("Couldn't process output as it is not an String or InputStream, instead it is: %s",
                                                         result.getOutput().getClass()));
    }
    return outputByteArray;
  }

  public static boolean containsIgnoreCase(String value, String predicate) {
    if (value == null || predicate == null) {
      return false;
    }

    return value.toLowerCase().contains(predicate.toLowerCase());
  }

  public static boolean isNotBlank(String v) {
    return !isBlank(v);
  }

  /**
   * <p>
   * Checks if a CharSequence is empty (""), null or whitespace only.
   * </p>
   *
   * <p>
   * Whitespace is defined by {@link Character#isWhitespace(char)}.
   * </p>
   *
   * <pre>
   * StringUtils.isBlank(null)      = true
   * StringUtils.isBlank("")        = true
   * StringUtils.isBlank(" ")       = true
   * StringUtils.isBlank("bob")     = false
   * StringUtils.isBlank("  bob  ") = false
   * </pre>
   *
   * @param cs the CharSequence to check, may be null
   * @return {@code true} if the CharSequence is null, empty or whitespace only
   */
  public static boolean isBlank(final CharSequence cs) {
    int strLen;
    if (cs == null || (strLen = cs.length()) == 0) {
      return true;
    }
    for (int i = 0; i < strLen; i++) {
      if (!isWhitespace(cs.charAt(i))) {
        return false;
      }
    }
    return true;
  }

  public static String readSchema(ClassLoader classLoader, String schemaPath) {
    try (InputStream in = classLoader.getResourceAsStream(schemaPath)) {
      if (in == null) {
        throw new IllegalArgumentException("Could not find schema at " + schemaPath);
      }
      return IOUtils.toString(in);
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  public static MetadataType loadXmlSchema(ClassLoader classLoader, String schemaPath, String qName) {
    schemaPath = makeRelativePath(schemaPath);
    String schemaContent = readSchema(classLoader, schemaPath);
    URL schemaURL = classLoader.getResource(schemaPath);
    SchemaCollector schemaCollector =
        getInstance().addSchema(schemaURL.toString(), new ByteArrayInputStream(schemaContent.getBytes()));

    Optional<MetadataType> metadataType = new XmlTypeLoader(schemaCollector).load(qName);

    if (!metadataType.isPresent()) {
      throw new RuntimeException("Could not load XML Schema " + schemaPath + " QName:" + qName);
    }

    return metadataType.get();
  }

  public static MetadataType loadJsonSchema(ClassLoader classLoader, String schemaPath, MetadataFormat metadataFormat) {
    schemaPath = makeRelativePath(schemaPath);
    String schemaContent = readSchema(classLoader, schemaPath);

    Optional<MetadataType> optionalMetadataType = new JsonTypeLoader(schemaContent).load(null);

    if (!optionalMetadataType.isPresent()) {
      throw new RuntimeException("Could not load Json Schema " + schemaPath);
    }

    MetadataType metadataType = optionalMetadataType.get();

    try {
      setMetadataFormat(metadataType, metadataFormat);
    } catch (NoSuchFieldException | IllegalAccessException e) {
      throw new RuntimeException("Could not set MetadataFormat to MetadataType", e);
    }

    return metadataType;
  }

  private static void setMetadataFormat(MetadataType metadataType, MetadataFormat newMetadataFormat)
      throws NoSuchFieldException, IllegalAccessException {
    Field metadataFormatField = BaseMetadataType.class.getDeclaredField("metadataFormat");
    metadataFormatField.setAccessible(true);
    metadataFormatField.set(metadataType, newMetadataFormat);
  }

  private static String makeRelativePath(String path) {
    return path.startsWith("/") ? path.substring(1) : path;
  }

  /**
   * Transform an iterator of TypedValue<?> into a list of TypedValue<String> consuming the iterator typed values using the
   * provided media type and charset.
   * 
   * @param iterator The iterator to consume
   * @param mediaType The primary source of the media type and charset. This is the media type for this specific iterator.
   * @param defaultMediaType Tee default source for the charset. The charset of this media type will be used if the primary one
   *        does not contain one.
   */
  public static List<TypedValue<String>> toList(Iterator<TypedValue<?>> iterator, MediaType mediaType,
                                                MediaType defaultMediaType) {
    List<TypedValue<String>> list = new ArrayList<>();

    Charset encoding = resolveCharset(of(mediaType), defaultMediaType);

    iterator.forEachRemaining(v -> {
      TypedValue<String> stringTypedValue;

      if (v.getDataType().getMediaType().getCharset().isPresent()
          && v.getDataType().getMediaType().getCharset().get().equals(encoding)) {
        stringTypedValue = consumeStringAndClose(v.getValue(), mediaType, encoding);
      } else {
        stringTypedValue =
            consumeStringTransformAndClose(v.getValue(), v.getDataType().getMediaType().getCharset().get(), mediaType, encoding);
      }

      list.add(stringTypedValue);
    });

    return list;
  }

  public static Charset resolveCharset(Optional<MediaType> mediaType, MediaType defaultMediaType) {
    return mediaType
        .flatMap(MediaType::getCharset)
        .orElseGet(() -> defaultMediaType.getCharset().orElse(defaultCharset()));
  }

  /**
   * Creates a typed value for an object, or returns null if the objeect is null.
   */
  public static TypedValue<?> getTypedValueOrNull(Object o) {
    return o != null ? TypedValue.of(o) : null;
  }

}
