/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.connection;

import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.concurrent.TimeUnit.SECONDS;
import static org.mule.runtime.api.connection.ConnectionValidationResult.success;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;
import static org.mule.runtime.extension.api.annotation.param.display.Placement.ADVANCED_TAB;
import static org.mule.runtime.http.api.HttpConstants.Protocol.HTTP;
import static org.mule.runtime.http.api.HttpConstants.Protocol.HTTPS;

import org.mule.runtime.api.connection.CachedConnectionProvider;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lifecycle.Startable;
import org.mule.runtime.api.lifecycle.Stoppable;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.api.tls.TlsContextFactoryBuilder;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.NullSafe;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.RefName;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.http.api.HttpConstants;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.client.auth.HttpAuthentication;
import org.mule.runtime.http.api.tcp.TcpClientSocketProperties;
import org.mule.runtime.http.api.tcp.TcpClientSocketPropertiesBuilder;
import org.mule.sdk.api.annotation.semantics.connectivity.ExcludeFromConnectivitySchema;

import com.mulesoft.connectivity.rest.commons.api.connection.validation.ConnectionValidationSettings;
import com.mulesoft.connectivity.rest.commons.api.proxy.HttpProxyConfig;
import com.mulesoft.connectivity.rest.commons.api.request.KeyValue;
import com.mulesoft.connectivity.rest.commons.api.request.QueryParam;
import com.mulesoft.connectivity.rest.commons.api.request.RequestHeader;
import com.mulesoft.connectivity.rest.commons.internal.RestConstants;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;

/**
 * Base class for Connection providers
 *
 * @since 1.0
 */
public abstract class BaseConnectionProvider implements CachedConnectionProvider<RestConnection>, Initialisable, Startable,
    Stoppable {

  /**
   * A mask pattern for building the name of the {@link HttpClient} that will be created to serve this provider.
   */
  static final String CLIENT_NAME_PATTERN = "rest.connect.%s";

  /**
   * The name of the config in which this connection is defined
   */
  @RefName
  private String configName;

  @Inject
  private HttpService httpService;

  @Inject
  private ExpressionLanguage expressionLanguage;

  /**
   * Default HTTP headers every request should include.
   */
  @Parameter
  @Optional
  @NullSafe
  @Placement(tab = ADVANCED_TAB, order = 1)
  @Expression(NOT_SUPPORTED)
  private List<RequestHeader> defaultHeaders;

  /**
   * Default Query parameters every request should include.
   */
  @Parameter
  @Optional
  @NullSafe
  @DisplayName("Query Parameters")
  @Placement(tab = ADVANCED_TAB, order = 2)
  @Expression(NOT_SUPPORTED)
  private List<QueryParam> defaultQueryParams;

  /**
   * The timeout for establishing connections to the remote service. This value is qualified by the {@link #connectionTimeoutUnit}
   */
  @Parameter
  @Optional(defaultValue = "30")
  @Expression(NOT_SUPPORTED)
  @Summary("The timeout for establishing connections to the remote service")
  @Placement(tab = ADVANCED_TAB, order = 3)
  private Integer connectionTimeout;

  /**
   * A time unit which qualifies the {@link #connectionTimeout}
   */
  @Parameter
  @Optional(defaultValue = "SECONDS")
  @Placement(tab = ADVANCED_TAB, order = 4)
  @Expression(NOT_SUPPORTED)
  @Summary("A time unit which qualifies the Connection Timeout")
  private TimeUnit connectionTimeoutUnit = SECONDS;

  /**
   * If false, each connection will be closed after the first request is completed.
   */
  @Parameter
  @Optional(defaultValue = "true")
  @Expression(NOT_SUPPORTED)
  @Placement(tab = ADVANCED_TAB, order = 5)
  @Summary("If false, each connection will be closed after the first request is completed.")
  @ExcludeFromConnectivitySchema
  private boolean usePersistentConnections = true;

  /**
   * The maximum number of outbound connections that will be kept open at the same time. By default the number of connections is
   * unlimited.
   */
  @Parameter
  @Optional(defaultValue = RestConstants.DEFAULT_MAX_CONNECTIONS)
  @Expression(NOT_SUPPORTED)
  @Placement(tab = ADVANCED_TAB, order = 6)
  @Summary("The maximum number of outbound connections that will be kept open at the same time")
  @ExcludeFromConnectivitySchema
  private Integer maxConnections;

  /**
   * A timeout for how long a connection can remain idle before it is closed. The value of this attribute is only used when
   * persistent connections are enabled. This value is qualified bY the {@link #connectionIdleTimeoutUnit}
   */
  @Parameter
  @Optional(defaultValue = RestConstants.DEFAULT_CONNECTION_IDLE_TIMEOUT)
  @Expression(NOT_SUPPORTED)
  @Placement(tab = ADVANCED_TAB, order = 7)
  @Summary("A timeout for how long a connection can remain idle before it is closed")
  private Integer connectionIdleTimeout;

  /**
   * A time unit which qualifies the {@link #connectionIdleTimeout}
   */
  @Parameter
  @Optional(defaultValue = "SECONDS")
  @Expression(NOT_SUPPORTED)
  @Placement(tab = ADVANCED_TAB, order = 8)
  @Summary("A time unit which qualifies the connection Idle Timeout")
  private TimeUnit connectionIdleTimeoutUnit = SECONDS;

  /**
   * Reusable configuration element for outbound connections through a proxy. A proxy element must define a host name and a port
   * attributes, and optionally can define a username and a password.
   */
  @Parameter
  @Optional
  @Summary("Reusable configuration element for outbound connections through a proxy")
  @Placement(tab = "Proxy")
  private HttpProxyConfig proxyConfig;

  /**
   * Whether or not received responses should be streamed, meaning processing will continue as soon as all headers are parsed and
   * the body streamed as it arrives. When enabled, the response MUST be eventually read since depending on the configured buffer
   * size it may not fit into memory and processing will stop until space is available.
   */
  @Parameter
  @Optional(defaultValue = "false")
  @Expression(NOT_SUPPORTED)
  @Placement(tab = ADVANCED_TAB, order = 9)
  @Summary("Whether or not received responses should be streamed")
  @ExcludeFromConnectivitySchema
  private boolean streamResponse = false;

  /**
   * The space in bytes for the buffer where the HTTP response will be stored.
   */
  @Parameter
  @Optional(defaultValue = RestConstants.DEFAULT_RESPONSE_BUFFER_SIZE)
  @Expression(NOT_SUPPORTED)
  @Placement(tab = ADVANCED_TAB, order = 10)
  @Summary("The space in bytes for the buffer where the HTTP response will be stored.")
  @ExcludeFromConnectivitySchema
  private int responseBufferSize;

  private final TlsContextFactoryBuilder defaultTlsContextFactoryBuilder = TlsContextFactory.builder();
  private TlsContextFactory effectiveTlsContextFactory = null;
  private HttpAuthentication authentication;

  private HttpClient httpClient;

  /**
   * @return The base uri that will be used for all HTTP requests.
   */
  public abstract String getBaseUri();

  @Override
  public final void initialise() throws InitialisationException {
    initialiseTls();
    verifyConnectionsParameters();

    authentication = buildAuthentication();
    if (authentication != null) {
      initialiseIfNeeded(authentication);
    }
  }

  /**
   * Starts the resources related to this connection provider.
   */
  @Override
  public void start() {
    startHttpClient();
  }

  /**
   * Stops the started resources related to this connection provider.
   */
  @Override
  public void stop() {
    httpClient.stop();
  }

  private void initialiseTls() throws InitialisationException {
    TlsParameterGroup tls = getTlsConfig().orElse(null);
    if (tls == null) {
      return;
    }

    final HttpConstants.Protocol protocol = tls.getProtocol();
    TlsContextFactory tlsContextFactory = tls.getTlsContext();

    if (protocol.equals(HTTP) && tlsContextFactory != null) {
      throw new InitialisationException(createStaticMessage("TlsContext cannot be configured with protocol HTTP, "
          + "when using tls:context you must set attribute protocol=\"HTTPS\""),
                                        this);
    }

    if (protocol.equals(HTTPS) && tlsContextFactory == null) {
      initialiseIfNeeded(defaultTlsContextFactoryBuilder);
      tlsContextFactory = defaultTlsContextFactoryBuilder.buildDefault();
    }
    if (tlsContextFactory != null) {
      initialiseIfNeeded(tlsContextFactory);
      effectiveTlsContextFactory = tlsContextFactory;
    }
  }

  @Override
  public final RestConnection connect() throws ConnectionException {
    try {
      return createConnection(httpClient,
                              authentication,
                              toMultiMap(defaultQueryParams),
                              toMultiMap(defaultHeaders));
    } catch (Exception e) {
      throw new ConnectionException("Could not create connection", e);
    }
  }

  /**
   * Initialises the http client that will be used by this connection provider to create new connections. Besides all the standard
   * configuration that is initialized here, the {@link #configureClient(HttpClientConfiguration.Builder)} method is provided in
   * order for extending classes to set it's custom configurations.
   */
  private void startHttpClient() {
    HttpClientConfiguration.Builder configuration = new HttpClientConfiguration.Builder()
        .setTlsContextFactory(effectiveTlsContextFactory)
        .setProxyConfig(proxyConfig)
        .setMaxConnections(maxConnections)
        .setUsePersistentConnections(usePersistentConnections)
        .setConnectionIdleTimeout(asMillis(connectionIdleTimeout, connectionTimeoutUnit))
        .setStreaming(streamResponse)
        .setResponseBufferSize(responseBufferSize)
        .setName(format(CLIENT_NAME_PATTERN, configName));

    configureTcpSocket(configuration);
    configureClient(configuration);

    httpClient = httpService.getClientFactory().create(configuration.build());
    httpClient.start();
  }

  private MultiMap<String, String> toMultiMap(Collection<? extends KeyValue> keyValues) {
    MultiMap<String, String> multiMap = new MultiMap<>();

    if (keyValues != null) {
      keyValues.forEach(kv -> multiMap.put(kv.getKey(), kv.getValue()));
    }

    return multiMap;
  }

  /**
   * Creates a new {@link RestConnection}
   *
   * @param httpClient the client to perform the requests
   * @param authentication a nullable authentication mechanism
   * @return a new {@link RestConnection}
   */
  protected RestConnection createConnection(HttpClient httpClient,
                                            HttpAuthentication authentication,
                                            MultiMap<String, String> defaultQueryParams,
                                            MultiMap<String, String> defaultHeaders) {
    return new DefaultRestConnection(getBaseUri(),
                                     getConfigName(),
                                     httpClient,
                                     authentication,
                                     defaultQueryParams,
                                     defaultHeaders, expressionLanguage);
  }

  /**
   * Implementations can override this method to specify the authentication mechanism to use. If not implemented, no authorization
   * will be used.
   *
   * @return an {@link HttpAuthentication} or {@code null}
   */
  protected HttpAuthentication buildAuthentication() {
    return null;
  }

  @Override
  public final void disconnect(RestConnection restConnection) {
    restConnection.stop();
  }

  @Override
  public ConnectionValidationResult validate(RestConnection restConnection) {
    return success();
  }

  protected ConnectionValidationResult validate(RestConnection restConnection,
                                                ConnectionValidationSettings settings) {
    return restConnection.validate(settings, asMillis(connectionTimeout, connectionTimeoutUnit));
  }

  /**
   * Template method that will be invoked just before the {@code httpConfiguration} is turned into an actual {@link HttpClient}.
   * It gives implementations a chance to do extra configurations.
   *
   * @param httpConfiguration the configuration builder for the {@link HttpClient} about to be created.
   */
  protected void configureClient(HttpClientConfiguration.Builder httpConfiguration) {
    // no-op by default
  }

  /**
   * TLS secured implementations <b>MUST</b> override this method to provide the {@link TlsParameterGroup} that configures TLS.
   *
   * @return an optional {@link TlsParameterGroup}
   */
  protected java.util.Optional<TlsParameterGroup> getTlsConfig() {
    return empty();
  }

  private void verifyConnectionsParameters() throws InitialisationException {

    if (getMaxConnections() == 0 || getMaxConnections() < RestConstants.UNLIMITED_CONNECTIONS) {
      throw new InitialisationException(createStaticMessage(
                                                            "The maxConnections parameter only allows positive values or -1 for unlimited concurrent connections."),
                                        this);
    }

    if (isUsePersistentConnections()) {
      connectionIdleTimeout = 0;
    }
  }

  private void configureTcpSocket(HttpClientConfiguration.Builder configuration) {
    TcpClientSocketPropertiesBuilder socketProperties = TcpClientSocketProperties.builder();
    if (connectionTimeout != null) {
      socketProperties.connectionTimeout(asMillis(connectionTimeout, connectionTimeoutUnit));
    }

    configuration.setClientSocketProperties(socketProperties.build());
  }

  private int asMillis(Integer value, TimeUnit unit) {
    if (value == null || value == -1) {
      return -1;
    }

    return Long.valueOf(unit.toMillis(value)).intValue();
  }

  protected String getConfigName() {
    return configName;
  }

  public List<RequestHeader> getDefaultHeaders() {
    return defaultHeaders;
  }

  public List<QueryParam> getDefaultQueryParams() {
    return defaultQueryParams;
  }

  protected Integer getConnectionTimeout() {
    return connectionTimeout;
  }

  protected TimeUnit getConnectionTimeoutUnit() {
    return connectionTimeoutUnit;
  }

  protected boolean isUsePersistentConnections() {
    return usePersistentConnections;
  }

  protected Integer getMaxConnections() {
    return maxConnections;
  }

  protected Integer getConnectionIdleTimeout() {
    return connectionIdleTimeout;
  }

  protected TimeUnit getConnectionIdleTimeoutUnit() {
    return connectionIdleTimeoutUnit;
  }

  protected HttpProxyConfig getProxyConfig() {
    return proxyConfig;
  }

  protected boolean isStreamResponse() {
    return streamResponse;
  }

  protected int getResponseBufferSize() {
    return responseBufferSize;
  }

  protected ExpressionLanguage getExpressionLanguage() {
    return expressionLanguage;
  }
}
