/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.model.builder.dataexpressions;

import com.mulesoft.connectivity.rest.commons.internal.model.builder.common.ArgumentBuilder;
import com.mulesoft.connectivity.rest.commons.internal.model.builder.common.ExpressionBuilder;
import com.mulesoft.connectivity.rest.commons.internal.model.common.Expression;
import com.mulesoft.connectivity.rest.commons.internal.model.dataexpressions.BindingArgument;
import com.mulesoft.connectivity.rest.commons.internal.model.dataexpressions.HttpRequestDataExpressionBinding;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * A fluent builder for http request data expressions.
 *
 * @since 1.0
 */
public class HttpRequestDataExpressionBindingBuilder {

  private final List<BindingArgumentBuilder> headers;
  private final List<BindingArgumentBuilder> uriParameters;
  private final List<BindingArgumentBuilder> queryParameters;
  private ExpressionBuilder bodyExpression;

  public HttpRequestDataExpressionBindingBuilder() {
    headers = new ArrayList<>();
    uriParameters = new ArrayList<>();
    queryParameters = new ArrayList<>();
  }

  private HttpRequestDataExpressionBindingBuilder addBindingArgument(String name, Consumer<BindingArgumentBuilder> consumer,
                                                                     List<BindingArgumentBuilder> argumentsList) {
    BindingArgumentBuilder bindingArgumentBuilder = new BindingArgumentBuilder(name);
    consumer.accept(bindingArgumentBuilder);
    argumentsList.add(bindingArgumentBuilder);
    return this;
  }

  public HttpRequestDataExpressionBindingBuilder header(String name, Consumer<BindingArgumentBuilder> consumer) {
    return addBindingArgument(name, consumer, headers);
  }

  public HttpRequestDataExpressionBindingBuilder uriParameter(String name, Consumer<BindingArgumentBuilder> consumer) {
    return addBindingArgument(name, consumer, uriParameters);
  }

  public HttpRequestDataExpressionBindingBuilder queryParameter(String name, Consumer<BindingArgumentBuilder> consumer) {
    return addBindingArgument(name, consumer, queryParameters);
  }

  public HttpRequestDataExpressionBindingBuilder body(Consumer<ExpressionBuilder> consumer) {
    ExpressionBuilder expressionBuilder = new ExpressionBuilder();
    consumer.accept(expressionBuilder);
    this.bodyExpression = expressionBuilder;
    return this;
  }

  private List<BindingArgument> buildArguments(List<BindingArgumentBuilder> arguments) {
    return arguments.stream().map(ArgumentBuilder::build).collect(Collectors.toList());
  }

  private List<BindingArgument> buildHeaders() {
    return buildArguments(headers);
  }

  private List<BindingArgument> buildUriParameters() {
    return buildArguments(uriParameters);
  }

  private List<BindingArgument> buildQueryParameters() {
    return buildArguments(queryParameters);
  }

  private Optional<Expression> buildBody() {
    return Optional.ofNullable(bodyExpression).map(ExpressionBuilder::build);
  }

  public HttpRequestDataExpressionBinding build() {
    return new HttpRequestDataExpressionBinding(buildHeaders(), buildUriParameters(), buildQueryParameters(),
                                                buildBody().orElse(null));

  }
}
