/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.metadatamodel.handler;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.IntersectionTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.builder.UnionTypeBuilder;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Stream;

import org.everit.json.schema.BooleanSchema;
import org.everit.json.schema.CombinedSchema;
import org.everit.json.schema.EnumSchema;
import org.everit.json.schema.NumberSchema;
import org.everit.json.schema.Schema;
import org.everit.json.schema.StringSchema;

public class CombinedHandler implements SchemaHandler {

  @Override
  public boolean handles(Schema schema) {
    return schema instanceof CombinedSchema;
  }

  @Override
  public TypeBuilder<?> handle(Schema schema, BaseTypeBuilder root, RestHandlerManager handlerManager,
                               ParsingContext parsingContext) {
    final CombinedSchema combinedSchema = (CombinedSchema) schema;
    final Collection<Schema> subschemas = combinedSchema.getSubschemas();
    if (combinedSchema.getCriterion() == CombinedSchema.ANY_CRITERION
        || combinedSchema.getCriterion() == CombinedSchema.ONE_CRITERION) {
      final UnionTypeBuilder unionType = root.unionType();
      for (Schema subschema : subschemas) {
        unionType.of(handlerManager.handle(subschema, parsingContext));
      }
      return unionType;
    } else if (combinedSchema.getCriterion() == CombinedSchema.ALL_CRITERION) {
      final TypeBuilder typeBuilder;
      ArrayList<Schema> uniqueSubschemas = new ArrayList<>();

      if (subschemas.size() == 2) {
        final boolean enumSchemaPresent =
            subschemas.stream().anyMatch(currentSubschema -> currentSubschema instanceof EnumSchema);
        final boolean stringSchemaPresent =
            subschemas.stream().anyMatch(currentSubschema -> currentSubschema instanceof StringSchema);
        // Only if "type": "string" and "enum" is defined we process enum schema, for the other
        // scenarios number/boolean/integer with enum we discard the enum schema and use the base type.
        for (Schema subschema : subschemas) {
          if ((enumSchemaPresent && stringSchemaPresent && subschema instanceof EnumSchema) ||
              (enumSchemaPresent && !stringSchemaPresent && !(subschema instanceof EnumSchema))) {
            uniqueSubschemas.add(subschema);
          }
        }
        // If we don't have the expected combination of types with enum just add every subschema to build an intersection
        if (uniqueSubschemas.isEmpty()) {
          uniqueSubschemas.addAll(subschemas);
        }
      } else {
        uniqueSubschemas.addAll(subschemas);
      }

      if (uniqueSubschemas.size() == 1) {
        typeBuilder = handlerManager.handle(uniqueSubschemas.iterator().next(), parsingContext);
      } else if (uniqueSubschemas.size() == 0) {
        typeBuilder = root.anyType();
      } else {
        final IntersectionTypeBuilder intersectionType = root.intersectionType();
        for (Schema subschema : uniqueSubschemas) {
          intersectionType.of(handlerManager.handle(subschema, parsingContext));
        }
        typeBuilder = intersectionType;
      }
      return typeBuilder;
    }
    return root.anyType();
  }
}
