/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.model.dataexpressions;

import static com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext.PROPERTY_CONFIGURATION;
import static com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext.PROPERTY_CONNECTION;
import static com.mulesoft.connectivity.rest.commons.internal.util.RequestStreamingUtils.doRequestAndConsumeString;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.stringValue;
import static java.lang.String.format;

import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.http.api.HttpConstants;

import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext;
import com.mulesoft.connectivity.rest.commons.internal.model.common.Expression;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * An evaluable -executable- HTTP Request, with its corresponding path, method and bindings. It will use the provided context to
 * get the configuration and connection that will allow making the actual HTTP Request.
 *
 * @since 1.0
 */
public class HttpRequestDataExpression implements DataExpression {

  private final String path;
  private final String method;
  private final HttpRequestDataExpressionBinding bindings;

  public HttpRequestDataExpression(String path, String method, HttpRequestDataExpressionBinding bindings) {
    this.path = path;
    this.method = method;
    this.bindings = bindings;
  }

  @Override
  public Object evaluate(EvaluationContext evaluationContext) {

    RestConfiguration configuration =
        (RestConfiguration) evaluationContext.resolveProperty(PROPERTY_CONFIGURATION);
    RestConnection connection = (RestConnection) evaluationContext.resolveProperty(PROPERTY_CONNECTION);

    final RestRequestBuilder restRequestBuilder = getRestRequestBuilder(evaluationContext);

    return doRequestAndConsumeString(connection, configuration, restRequestBuilder, getDefaultResponseMediaType());
  }

  private MediaType getDefaultResponseMediaType() {
    return MediaType.APPLICATION_JSON;
  }

  private String getPathTemplate() {
    return path;
  }

  private Optional<HttpRequestDataExpressionBinding> getBindings() {
    return Optional.ofNullable(bindings);
  }

  private RestRequestBuilder getRestRequestBuilder(EvaluationContext evaluationContext) {
    final String path = buildRequestPath(evaluationContext);
    RestConfiguration configuration =
        (RestConfiguration) evaluationContext.resolveProperty(PROPERTY_CONFIGURATION);
    RestConnection connection = (RestConnection) evaluationContext.resolveProperty(PROPERTY_CONNECTION);
    final RestRequestBuilder restRequestBuilder =
        new RestRequestBuilder(connection.getBaseUri(), path, HttpConstants.Method.valueOf(method.toUpperCase()));

    getBindings().ifPresent(httpRequestDataExpressionBinding -> {
      httpRequestDataExpressionBinding.getHeaders().forEach(header -> restRequestBuilder
          .addHeader(header.getName(), getParameterValue(header.getValue(), evaluationContext)));
      httpRequestDataExpressionBinding.getQueryParameters().forEach(queryParameter -> restRequestBuilder
          .addQueryParam(queryParameter.getName(), getParameterValue(queryParameter.getValue(), evaluationContext)));
    });

    return restRequestBuilder;
  }

  private String buildRequestPath(EvaluationContext evaluationContext) {
    if (bindings.getUriParameters().isEmpty()) {
      return getPathTemplate();
    }

    Map<String, String> uriParamValues = new HashMap<>();

    getBindings().ifPresent(httpRequestDataExpressionBinding -> httpRequestDataExpressionBinding.getUriParameters()
        .forEach(uriParameter -> uriParamValues
            .put(uriParameter.getName(), getParameterValue(uriParameter.getValue(), evaluationContext))));

    String path = getPathTemplate();
    for (String key : uriParamValues.keySet()) {
      path = path.replace("{" + key + "}", uriParamValues.get(key));
    }

    return path;
  }

  private String getParameterValue(Expression expression, EvaluationContext evaluationContext) {
    Object result = expression.evaluate(evaluationContext);
    if (result instanceof TypedValue) {
      TypedValue<?> typedValue = (TypedValue<?>) result;
      result = typedValue.getValue();
    }

    // TODO: RSDK-298: Which types can be converted to string?
    if (result instanceof String | result instanceof Integer | result instanceof Long | result instanceof Double) {
      return stringValue(result);
    } else {
      // TODO: RSDK-298: In which cases should this break?
      throw new IllegalArgumentException(format("Expression '%s' did not return a String but `%s`",
                                                expression,
                                                result != null ? result.getClass().getSimpleName() : "null"));
    }
  }
}
