/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.model.resolvers;

import com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext;
import com.mulesoft.connectivity.rest.commons.internal.model.common.SimpleEvaluationContext;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * A reference to a resolver declaration. Includes the name of the resolver declaration this points to and provides arguments for
 * its parameters.
 *
 * @since 1.0
 */
public class ResolverReference implements ResolverExpression {

  private final static String KEY_PARAMETER = "parameter";

  private final String declarationName;
  private final Map<String, ResolverArgument> argumentsByName;

  public ResolverReference(String declarationName, List<ResolverArgument> arguments) {
    this.declarationName = declarationName;

    argumentsByName = new HashMap<>();
    arguments.forEach(resolverArgument -> argumentsByName.put(resolverArgument.getName(), resolverArgument));
  }

  /**
   * Looks for the referenced resolver declaration in the global context and evaluates it using the provided evaluation context.
   *
   * @param evaluationContext The evaluation context that will be used to evaluate the resolver declaration. It should contain
   *        arguments for each of the declaration's parameters.
   *
   * @return The result of evaluating the resolver declaration against the provided context.
   */
  @Override
  public Object evaluate(EvaluationContext evaluationContext) {
    final SimpleEvaluationContext functionEvaluationContext =
        new SimpleEvaluationContext(evaluationContext.getGlobalEvaluationContext().orElse(null));
    final ResolverDeclaration<?> declaration = evaluationContext.resolveDeclaration(declarationName);

    Map<String, Object> parametersMap = new HashMap<>();
    declaration.getParameters().forEach(resolverParameter -> {
      final ResolverArgument resolverArgument =
          Optional.ofNullable(argumentsByName.get(resolverParameter.getName())).orElseThrow(() -> new RuntimeException(String
              .format("No argument provided for parameter %s", resolverParameter.getName())));
      parametersMap.put(resolverParameter.getName(), resolverArgument.getValue().evaluate(evaluationContext));
    });

    functionEvaluationContext.define(KEY_PARAMETER, parametersMap);
    return declaration.evaluate(functionEvaluationContext);
  }
}
