/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.connection;

import static com.mulesoft.connectivity.rest.commons.api.connection.validation.ConnectionValidator.validateConnectionResponse;
import static com.mulesoft.connectivity.rest.commons.api.error.RestError.CONNECTIVITY;
import static com.mulesoft.connectivity.rest.commons.api.error.RestError.TIMEOUT;
import static com.mulesoft.connectivity.rest.commons.api.error.RestError.getErrorByCode;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.PAYLOAD_VAR;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.closeStream;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.containsIgnoreCase;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.getMediaType;
import static java.lang.String.format;
import static java.util.Collections.emptyMap;
import static java.util.Objects.requireNonNull;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JAVA;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;

import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.client.auth.HttpAuthentication;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.multipart.HttpPart;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.connection.validation.ConnectionValidationSettings;
import com.mulesoft.connectivity.rest.commons.api.error.RequestException;
import com.mulesoft.connectivity.rest.commons.api.error.RestError;
import com.mulesoft.connectivity.rest.commons.api.interception.HttpResponseInterceptor;
import com.mulesoft.connectivity.rest.commons.api.interception.InterceptionHttpRequest;
import com.mulesoft.connectivity.rest.commons.api.interception.RestErrorHttpResponseInterceptor;
import com.mulesoft.connectivity.rest.commons.api.operation.EntityRequestParameters;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.internal.interception.DefaultInterceptionHttpRequest;
import com.mulesoft.connectivity.rest.commons.internal.model.http.HttpEntityCursorStreamProviderBased;
import com.mulesoft.connectivity.rest.commons.internal.multipart.DWMultipartPayloadBuilder;
import com.mulesoft.connectivity.rest.commons.internal.util.ConnectionValidationUtils;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeoutException;
import java.util.function.BiConsumer;
import java.util.function.Predicate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default implementation of {@link RestConnection}.
 *
 * @since 1.0
 */
public class DefaultRestConnection implements RestConnection {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultRestConnection.class);
  private static final String REMOTELY_CLOSED = "Remotely closed";
  private static final String BOUNDARY = "boundary";

  private final String baseUri;
  private final String configName;
  private final HttpClient httpClient;
  private final HttpAuthentication authentication;
  private final MultiMap<String, String> defaultQueryParams;
  private final MultiMap<String, String> defaultHeaders;
  private final ExpressionLanguage expressionLanguage;
  private final Map<String, Object> bindings;

  /**
   * Creates a new instance
   *
   * @param baseUri the service base uri
   * @param configName the name of the config that owns this connection
   * @param httpClient the client to use
   * @param authentication the authentication mechanism to use, or {@code null}
   * @param defaultQueryParams query params to be automatically added to all requests done through this connection
   * @param defaultHeaders headers to be automatically added to all requests done through this connection
   * @param expressionLanguage expression language support
   */
  public DefaultRestConnection(String baseUri,
                               String configName,
                               HttpClient httpClient,
                               HttpAuthentication authentication,
                               MultiMap<String, String> defaultQueryParams,
                               MultiMap<String, String> defaultHeaders, ExpressionLanguage expressionLanguage) {
    this(baseUri, configName, httpClient, authentication, defaultQueryParams, defaultHeaders, expressionLanguage, emptyMap());
  }

  /**
   * Creates a new instance
   *
   * @param baseUri the service base uri
   * @param configName the name of the config that owns this connection
   * @param httpClient the client to use
   * @param authentication the authentication mechanism to use, or {@code null}
   * @param defaultQueryParams query params to be automatically added to all requests done through this connection
   * @param defaultHeaders headers to be automatically added to all requests done through this connection
   * @param expressionLanguage expression language support
   * @param bindings parameters added to this connection
   */
  public DefaultRestConnection(String baseUri,
                               String configName,
                               HttpClient httpClient,
                               HttpAuthentication authentication,
                               MultiMap<String, String> defaultQueryParams,
                               MultiMap<String, String> defaultHeaders, ExpressionLanguage expressionLanguage,
                               Map<String, Object> bindings) {
    requireNonNull(expressionLanguage, "'expressionLanguage' cannot be null");

    this.baseUri = baseUri;
    this.configName = configName;
    this.httpClient = httpClient;
    this.authentication = authentication;
    this.defaultQueryParams = nullSafe(defaultQueryParams);
    this.defaultHeaders = nullSafe(defaultHeaders);
    this.expressionLanguage = expressionLanguage;
    this.bindings = bindings;
  }

  private void merge(MultiMap<String, String> defaultValues,
                     Predicate<String> appendPredicate,
                     BiConsumer<String, List<String>> appender) {

    defaultValues.keySet().forEach(k -> {
      if (appendPredicate.test(k)) {
        appender.accept(k, defaultValues.getAll(k));
      }
    });
  }

  /**
   * {@inheritDoc}
   */
  public CompletableFuture<Result<String, HttpResponseAttributes>> bodylessRequest(RestRequestBuilder requestBuilder,
                                                                                   int responseTimeoutMillis,
                                                                                   MediaType defaultResponseMediaType,
                                                                                   StreamingHelper streamingHelper) {

    CompletableFuture<Result<String, HttpResponseAttributes>> future = new CompletableFuture<>();

    request(requestBuilder, responseTimeoutMillis, defaultResponseMediaType, streamingHelper)
        .whenComplete((r, e) -> {
          if (e != null) {
            future.completeExceptionally(e);
          } else {
            try {
              Result.Builder<String, HttpResponseAttributes> voidResult = Result.<String, HttpResponseAttributes>builder()
                  .output("")
                  .length(0);

              r.getAttributes().ifPresent(voidResult::attributes);
              r.getMediaType().ifPresent(voidResult::mediaType);
              r.getAttributesMediaType().ifPresent(voidResult::attributesMediaType);

              future.complete(voidResult.build());

            } finally {
              closeStream(r.getOutput());
            }
          }
        });

    return future;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CompletableFuture<Result<InputStream, HttpResponseAttributes>> request(RestRequestBuilder requestBuilder,
                                                                                int responseTimeoutMillis,
                                                                                MediaType defaultResponseMediaType,
                                                                                StreamingHelper streamingHelper) {

    CompletableFuture<Result<InputStream, HttpResponseAttributes>> future = new CompletableFuture<>();

    HttpRequest request = getHttpRequest(requestBuilder);
    try {
      httpClient.sendAsync(request, HttpRequestOptions.builder()
          .responseTimeout(responseTimeoutMillis)
          .followsRedirect(true)
          .authentication(authentication)
          .build()).whenComplete((response, t) -> {
            try {
              if (t != null) {
                handleRequestException(t, request, future);
              } else {
                handleResponse(response, request, requestBuilder.getUriParams(), defaultResponseMediaType, future,
                               streamingHelper,
                               requestBuilder.getResponseInterceptor());
              }
            } catch (MuleRuntimeException e) {
              future.completeExceptionally(e);
            } catch (Throwable e) {
              future.completeExceptionally(new MuleRuntimeException(
                                                                    createStaticMessage("Unhandled exception on completing send async."),
                                                                    e));
            }
          });
    } catch (Throwable t) {
      handleRequestException(t, request, future);
    }

    return future;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectionValidationResult validate(ConnectionValidationSettings settings, int responseTimeoutMillis) {

    RestRequestBuilder requestBuilder = new RestRequestBuilder(getBaseUri(),
                                                               settings.getTestConnectionPath(),
                                                               settings.getHttpMethod(),
                                                               new EntityRequestParameters());
    HttpResponse response;

    try {
      response = httpClient.send(getHttpRequest(requestBuilder),
                                 HttpRequestOptions.builder()
                                     .responseTimeout(responseTimeoutMillis)
                                     .followsRedirect(true)
                                     .authentication(authentication)
                                     .build());
    } catch (TimeoutException e) {
      return ConnectionValidationUtils.connectionValidationResult(new ModuleException(e.getMessage(), TIMEOUT, e));
    } catch (Throwable t) {
      return ConnectionValidationUtils.connectionValidationResult(new ModuleException(t.getMessage(), CONNECTIVITY, t));
    }

    return validateConnectionResponse(response, settings);
  }

  private HttpRequest getHttpRequest(RestRequestBuilder requestBuilder) {
    final MultiMap<String, String> headers = requestBuilder.getHeaders();
    final MultiMap<String, String> queryParams = requestBuilder.getQueryParams();

    merge(defaultHeaders, k -> !headers.containsKey(k), requestBuilder::addHeaders);
    merge(defaultQueryParams, k -> !queryParams.containsKey(k), requestBuilder::addQueryParams);

    return buildRequest(requestBuilder);
  }

  private void handleResponse(HttpResponse response,
                              HttpRequest request,
                              Map<String, String> uriParams,
                              MediaType defaultResponseMediaType,
                              CompletableFuture<Result<InputStream, HttpResponseAttributes>> future,
                              StreamingHelper streamingHelper, HttpResponseInterceptor onHttpResponseInterceptor) {
    response = postProcessResponse(response, request, uriParams, onHttpResponseInterceptor);

    RestError error = getErrorByCode(response.getStatusCode()).orElse(null);

    if (error != null) {
      handleResponseError(response, defaultResponseMediaType, future, streamingHelper, error);
    } else {
      future.complete(toResult(response, false, defaultResponseMediaType, streamingHelper));
    }
  }

  private HttpResponse postProcessResponse(HttpResponse response, HttpRequest request, Map<String, String> uriParams,
                                           HttpResponseInterceptor httpResponseInterceptor) {
    InterceptionHttpRequest interceptionHttpRequest = new DefaultInterceptionHttpRequest(request, uriParams);

    response = HttpResponse.builder()
        .statusCode(response.getStatusCode())
        .reasonPhrase(response.getReasonPhrase())
        .headers(response.getHeaders())
        .entity(toRepeatableStreamingHttpEntity(response))
        .build();

    try {
      if (httpResponseInterceptor.match(interceptionHttpRequest, response, expressionLanguage)) {
        return httpResponseInterceptor.intercept(interceptionHttpRequest, response, expressionLanguage);
      }

      // Use SDK default error response interceptor
      RestErrorHttpResponseInterceptor restErrorResponseInterceptor = RestErrorHttpResponseInterceptor.getInstance();
      if (restErrorResponseInterceptor.match(interceptionHttpRequest, response, expressionLanguage)) {
        return restErrorResponseInterceptor.intercept(interceptionHttpRequest, response, expressionLanguage);
      }
    } catch (Exception e) {
      throw new MuleRuntimeException(createStaticMessage("There was an error while executing the HttpResponseInterceptor for the server's response"),
                                     e);
    }

    return response;
  }

  /**
   * Creates an {@link HttpEntity} that is backed up with a {@link CursorStreamProvider} to allow streaming the entity content.
   */
  private HttpEntity toRepeatableStreamingHttpEntity(HttpResponse response) {
    HttpEntity entity = response.getEntity();

    if (entity.isComposed()) {
      try {
        DWMultipartPayloadBuilder builder = new DWMultipartPayloadBuilder(expressionLanguage);

        if (!response.getHeaders().containsKey(CONTENT_TYPE)) {
          throw new MuleRuntimeException(createStaticMessage(format("Missing %s header to process multipart HTTP Response",
                                                                    CONTENT_TYPE)));
        }
        MediaType mediaType = MediaType.parse(response.getHeaders().get(CONTENT_TYPE));
        String boundary = mediaType.getParameter(BOUNDARY);

        if (boundary != null) {
          builder.setBoundary(boundary);
        }

        for (HttpPart httpPart : entity.getParts()) {
          MediaType contentType = MediaType.parse(httpPart.getContentType());
          DataType dataType = DataType.builder().mediaType(contentType.toRfcString()).build();
          TypedValue<InputStream> typedValue = new TypedValue(httpPart.getInputStream(), dataType);

          if (httpPart.getFileName() != null) {
            builder.addFilePart(httpPart.getName(), httpPart.getFileName(), typedValue);
          } else {
            builder.addPart(httpPart.getName(), typedValue);
          }
        }
        return new HttpEntityCursorStreamProviderBased(builder.asCursorStreamProvider().getValue(), entity.getBytesLength());
      } catch (IOException e) {
        throw new MuleRuntimeException(createStaticMessage("There was an error while building a repeatable stream for a composed HttpEntity"),
                                       e);
      }
    } else {
      TypedValue<CursorStreamProvider> streamProviderTypedValue = (TypedValue<CursorStreamProvider>) expressionLanguage
          .evaluate(PAYLOAD_VAR,
                    DataType.CURSOR_STREAM_PROVIDER,
                    BindingContext.builder()
                        .addBinding(PAYLOAD_VAR, TypedValue.of(entity.getContent()))
                        .build());
      return new HttpEntityCursorStreamProviderBased(streamProviderTypedValue.getValue(), entity.getBytesLength());
    }
  }

  protected void handleResponseError(HttpResponse response,
                                     MediaType defaultResponseMediaType,
                                     CompletableFuture<Result<InputStream, HttpResponseAttributes>> future,
                                     StreamingHelper streamingHelper,
                                     RestError error) {
    future
        .completeExceptionally(new RequestException(error, toResult(response, true, defaultResponseMediaType, streamingHelper)));
  }

  private void handleRequestException(Throwable t,
                                      HttpRequest request,
                                      CompletableFuture<Result<InputStream, HttpResponseAttributes>> future) {

    checkIfRemotelyClosed(t, request);
    RestError error = t instanceof TimeoutException ? TIMEOUT : CONNECTIVITY;
    future.completeExceptionally(new ModuleException(t.getMessage(), error, t));
  }

  protected <T> Result<T, HttpResponseAttributes> toResult(HttpResponse response,
                                                           boolean isError,
                                                           MediaType defaultResponseMediaType,
                                                           StreamingHelper streamingHelper) {
    Result.Builder<T, HttpResponseAttributes> builder = Result.builder();

    HttpEntity entity = response.getEntity();
    Object content = entity.getContent();
    if (isError) {
      content = streamingHelper != null ? streamingHelper.resolveCursorProvider(content) : content;
    }

    builder.output((T) content);
    entity.getBytesLength().ifPresent(builder::length);

    MediaType contentType = getMediaType(response, defaultResponseMediaType);

    builder.mediaType(contentType);
    builder.attributes(toAttributes(response)).attributesMediaType(APPLICATION_JAVA);

    return builder.build();
  }

  protected HttpRequest buildRequest(RestRequestBuilder requestBuilder) {
    return requestBuilder.build();
  }

  protected HttpResponseAttributes toAttributes(HttpResponse response) {
    return new HttpResponseAttributes(response.getStatusCode(), response.getReasonPhrase(), response.getHeaders());
  }

  /**
   * Template method invoked by the {@link #stop()} to perform custom actions before the actual stopping takes place. This method
   * should not fail, but if it does, it won't stop the actual stopping process
   */
  protected void beforeStop() {
    // default implementation is empty
  }

  /**
   * Template method invoked by the {@link #stop()} to perform custom actions after the actual stopping takes place. This method
   * should not fail, but if it does, it won't stop the actual stopping process
   */
  protected void afterStop() {
    // default implementation is empty
  }

  private void checkIfRemotelyClosed(Throwable exception, HttpRequest request) {
    if ("https".equals(request.getUri().getScheme()) && containsIgnoreCase(exception.getMessage(), REMOTELY_CLOSED)) {
      LOGGER.error("Remote host closed connection. Possible SSL/TLS handshake issue. Check protocols, cipher suites and "
          + "certificate set up. Use -Djavax.net.debug=ssl for further debugging.");
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getBaseUri() {
    return baseUri;
  }

  /**
   * Stops this connection by invoking {@link #beforeStop()} and {@link #afterStop()}, which allow implementors to customize the
   * stopping process. Notice that those methods should not fail, but if they do, they won't stop resources from being freed.
   */
  @Override
  public final void stop() {
    try {
      beforeStop();
    } catch (Throwable t) {
      LOGGER.warn(format("Exception found before stopping config '%s'", configName), t);
    }

    try {
      afterStop();
    } catch (Throwable t) {
      LOGGER.warn(format("Exception found after stopping config '%s'", configName), t);
    }
  }

  /**
   * @return the custom bindings added to the connection or an empty map
   */
  @Override
  public Map<String, Object> getBindings() {
    return bindings;
  }

  private MultiMap<String, String> nullSafe(MultiMap<String, String> multiMap) {
    return multiMap != null ? multiMap : new MultiMap<>();
  }

  protected ExpressionLanguage getExpressionLanguage() {
    return expressionLanguage;
  }

}
