/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.error;

import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JAVA;

import org.mule.runtime.api.exception.ErrorMessageAwareException;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.operation.Result;

import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;

/**
 * {@link ModuleException} to be raised when a request fails.
 * <p>
 * Because the HTTP protocol also emits responses when the request fails, this exception carries the error response by
 * implementing the {@link ErrorMessageAwareException}. The response is accessible through the {@link #getErrorMessage()} method.
 *
 * @since 1.0
 */
public class RequestException extends ModuleException implements ErrorMessageAwareException {

  private final Message message;

  /**
   * Creates a new instance
   *
   * @param error the error type
   * @param response the obtained response
   */
  public RequestException(RestError error, Result<Object, HttpResponseAttributes> response) {
    super(response.getAttributes().get().getReasonPhrase(), error);
    Message.Builder builder = Message.builder()
        .payload(getPayloadTypedValue(response))
        .mediaType(getMediaType(response));

    setAttributes(response, builder);
    message = builder.build();
  }


  private TypedValue<Object> getPayloadTypedValue(Result<Object, HttpResponseAttributes> response) {
    DataType dataType = DataType.builder()
        .type(response.getOutput().getClass())
        .mediaType(getMediaType(response))
        .build();

    return new TypedValue<>(response.getOutput(), dataType, response.getByteLength());
  }

  private void setAttributes(Result<Object, HttpResponseAttributes> response, Message.Builder builder) {
    response.getAttributes().ifPresent(attributes -> {
      DataType dataType = DataType.builder()
          .type(attributes.getClass())
          .mediaType(response.getAttributesMediaType().orElse(APPLICATION_JAVA))
          .build();

      builder.attributes(new TypedValue<>(attributes, dataType));
    });

  }

  private MediaType getMediaType(Result<Object, HttpResponseAttributes> response) {
    return response.getMediaType().orElse(null);
  }

  @Override
  public Message getErrorMessage() {
    return message;
  }
}
