/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.interception;

import static com.mulesoft.connectivity.rest.commons.api.interception.CompositeHttpResponseInterceptor.ExecutionStrategy.ONLY_FIRST;
import static java.util.Objects.requireNonNull;

import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.util.List;

/**
 * A composite implementation for {@link HttpResponseInterceptor} that executes the first matching {@link HttpResponseInterceptor}
 * from the {@link List} or all of them depending on the execution strategy provided.
 *
 * @since 1.0
 */
public class CompositeHttpResponseInterceptor implements HttpResponseInterceptor {

  private List<HttpResponseInterceptor> httpResponseInterceptors;
  private ExecutionStrategy executionStrategy;

  public CompositeHttpResponseInterceptor(List<HttpResponseInterceptor> httpResponseInterceptors,
                                          ExecutionStrategy executionStrategy) {
    requireNonNull(httpResponseInterceptors);
    requireNonNull(executionStrategy);

    this.httpResponseInterceptors = httpResponseInterceptors;
    this.executionStrategy = executionStrategy;
  }

  @Override
  public boolean match(InterceptionHttpRequest httpRequest, HttpResponse httpResponse, ExpressionLanguage expressionLanguage) {
    return httpResponseInterceptors.stream()
        .filter(responseInterceptor -> responseInterceptor.match(httpRequest, httpResponse, expressionLanguage))
        .findAny().isPresent();
  }

  @Override
  public HttpResponse intercept(InterceptionHttpRequest httpRequest, HttpResponse httpResponse,
                                ExpressionLanguage expressionLanguage) {
    for (HttpResponseInterceptor httpResponseInterceptor : httpResponseInterceptors) {
      if (httpResponseInterceptor.match(httpRequest, httpResponse, expressionLanguage)) {
        httpResponse = httpResponseInterceptor.intercept(httpRequest, httpResponse, expressionLanguage);
        if (ONLY_FIRST == executionStrategy) {
          return httpResponse;
        }
      }
    }
    if (ONLY_FIRST == executionStrategy) {
      throw new IllegalStateException(
                                      "At least one responseInterceptor from the composition should have matched the HttpResponse to be processed");
    } else {
      return httpResponse;
    }
  }

  /**
   * Defines if {@link #ALL} or {@link #ONLY_FIRST} of the interceptors should be executed by this composition. If
   * {@link #ONLY_FIRST} is selected the first interceptor that matches the condition is executed and next are skipped.
   * {@link #ALL} every interceptor is executed and input is invoked with the output of the previous, if matching condition is
   * satisfied.
   */
  public enum ExecutionStrategy {
    ALL, ONLY_FIRST
  }
}
