/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation;

import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.CONTEXT_KEY_PARAMETERS;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.PAYLOAD_VAR;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.CONNECTION;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.CONFIG;
import static com.mulesoft.connectivity.rest.commons.internal.util.DwUtils.isExpression;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.consumeStringAndClose;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.getTypedValueOrNull;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.resolveCharset;
import static java.util.Optional.empty;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.DataType.STRING;
import static org.mule.runtime.api.metadata.MediaType.ANY;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JSON;
import static org.mule.runtime.core.api.util.IOUtils.closeQuietly;
import static org.mule.runtime.api.el.BindingContext.builder;

import com.mulesoft.connectivity.rest.commons.api.interception.HttpResponseInterceptor;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.HttpConstants;

import com.mulesoft.connectivity.rest.commons.api.binding.HttpRequestBinding;
import com.mulesoft.connectivity.rest.commons.api.binding.HttpResponseBinding;
import com.mulesoft.connectivity.rest.commons.api.binding.ParameterBinding;
import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.multipart.MultipartPayloadBuilder;
import com.mulesoft.connectivity.rest.commons.internal.multipart.DWMultipartPayloadBuilder;
import com.mulesoft.connectivity.rest.commons.internal.util.CloserCompletionCallbackDecorator;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import javax.inject.Inject;

/**
 * Base class for operations that consume a remote REST endpoint
 *
 * @since 1.0
 */
public abstract class BaseRestOperation {

  @Inject
  private ExpressionLanguage expressionLanguage;

  /**
   * Default constructor used by Mule/SDK to instantiate this operation.
   */
  public BaseRestOperation() {

  }

  /**
   * Constructor to allow creating an operation without using IoC from Mule/SDK. This is useful when reusing connector's
   * operations.
   * 
   * @param expressionLanguage
   */
  public BaseRestOperation(ExpressionLanguage expressionLanguage) {
    this.expressionLanguage = expressionLanguage;
  }

  /**
   * Performs the request using the given {@code builder} and {@code connection} and completes the {@code callback} accordingly.
   *
   * @param config the operation's config
   * @param connection the connection to use
   * @param builder the request builder
   * @param responseTimeoutMillis the request timeout in millis
   * @param streamingHelper the {@link StreamingHelper}
   * @param callback the operation's {@link CompletionCallback}
   */
  protected void doRequest(RestConfiguration config,
                           RestConnection connection,
                           RestRequestBuilder builder,
                           int responseTimeoutMillis,
                           StreamingHelper streamingHelper,
                           Map<String, Object> parameterBindings,
                           CompletionCallback<InputStream, HttpResponseAttributes> callback) {
    try {
      connection.request(builder, responseTimeoutMillis, resolveDefaultResponseMediaType(config), streamingHelper)
          .whenComplete(handleResponse(callback, parameterBindings));
    } catch (Throwable t) {
      callback.error(t);
    }
  }

  protected MediaType resolveDefaultResponseMediaType(RestConfiguration config) {
    MediaType mediaType = getDefaultResponseMediaType();
    if (!mediaType.getCharset().isPresent()) {
      mediaType = mediaType.withCharset(config.getCharset());
    }

    return mediaType;
  }

  /**
   * Performs the request using the given {@code builder} and {@code connection} and completes the {@code callback} accordingly,
   * expecting the response to not contain a body.
   * <p>
   * If successful, the callback will be completed with an empty String as payload.
   *
   * @param connection the connection to use
   * @param builder the request builder
   * @param responseTimeoutMillis the request timeout in millis
   * @param streamingHelper the {@link StreamingHelper}
   * @param callback the operation's {@link CompletionCallback}
   */
  protected void doVoidRequest(RestConnection connection,
                               RestRequestBuilder builder,
                               int responseTimeoutMillis,
                               StreamingHelper streamingHelper,
                               Map<String, Object> parameterBindings,
                               CompletionCallback<String, HttpResponseAttributes> callback) {
    try {
      connection.bodylessRequest(builder, responseTimeoutMillis, ANY, streamingHelper)
          .whenComplete(handleResponse(callback, parameterBindings));
    } catch (Throwable t) {
      callback.error(t);
    }
  }

  protected <T> BiConsumer<Result<T, HttpResponseAttributes>, Throwable> handleResponse(CompletionCallback<T, HttpResponseAttributes> callback,
                                                                                        Map<String, Object> parameterBindings) {

    return (result, error) -> {
      if (error != null) {
        callback.error(error);
      } else {

        try {
          // If output is not an input stream, this is a void operation, we have no post processing to do.
          // This is valid because we are only post processing the payload.
          if (result.getOutput() instanceof InputStream) {
            result = (Result<T, HttpResponseAttributes>) postProcessResult((Result<InputStream, HttpResponseAttributes>) result,
                                                                           parameterBindings);
          }
        } catch (Throwable t) {
          callback.error(t);
        }

        callback.success(result);
      }
    };
  }

  /**
   * Allows to make a multipart request by facilitating the construction of the multipart body through a
   * {@link MultipartPayloadBuilder}.
   * <p>
   * This operation <b>MUST NOT</b> be completed using the {@code callback} passed as a parameter, but the one provided through
   * the {@code consumer}
   *
   * @param builderConfigurer a {@link Consumer} which gives access to the {@link MultipartPayloadBuilder} and allows to configure
   *        it
   * @param callback the operation's {@link CompletionCallback}
   * @param consumer the {@link Consumer} which gives access to the built multipart body and the effective callback that should
   *        complete this operation
   * @param <T> the generic type of the {@code callback} payload type
   * @param <A> the generic type of the {@code callback} attributes type
   */
  protected <T, A> void withMultipart(Consumer<MultipartPayloadBuilder> builderConfigurer,
                                      CompletionCallback<T, A> callback,
                                      BiConsumer<TypedValue<InputStream>, CompletionCallback<T, A>> consumer) {

    MultipartPayloadBuilder builder = new DWMultipartPayloadBuilder(expressionLanguage);
    builderConfigurer.accept(builder);

    TypedValue<InputStream> multipart = builder.build();
    try {
      consumer.accept(multipart, new CloserCompletionCallbackDecorator<>(callback, multipart.getValue()));
    } catch (Exception e) {
      closeQuietly(multipart.getValue());
      callback.error(e);
    }
  }

  /**
   * This method specifies the {@link MediaType} that should be assumed the response to have in case the remote service doesn't
   * specify a {@code Content-Type} header.
   * <p>
   * By default, this method returns {@link MediaType#APPLICATION_JSON} since it's the most common response type. However, this
   * method should be overwritten if a different type of media type is to be expected, or if you know that a certain encoding will
   * be enforced.
   *
   * @return the {@link MediaType} to assign the response in case the server doesn't specify one in its response
   */
  protected MediaType getDefaultResponseMediaType() {
    return APPLICATION_JSON;
  }

  protected ExpressionLanguage getExpressionLanguage() {
    return expressionLanguage;
  }

  protected RestRequestBuilder getRequestBuilderWithBindings(String baseUri, String path, HttpConstants.Method method,
                                                             RequestParameters requestParameters,
                                                             ConfigurationOverrides overrides,
                                                             RestConnection connection,
                                                             RestConfiguration config,
                                                             Map<String, Object> parameterBindings) {
    BindingContext bindingContext = getRequestBindingContext(connection, config, parameterBindings);
    HttpRequestBinding requestBindings = getRequestBindings();

    RestRequestBuilder restRequestBuilder = new RestRequestBuilder(baseUri, path, method, requestParameters);

    for (ParameterBinding binding : requestBindings.getUriParams()) {
      restRequestBuilder.addUriParam(binding.getKey(), resolveParameterExpression(binding.getValue(), bindingContext));
    }

    for (ParameterBinding binding : requestBindings.getQueryParams()) {
      restRequestBuilder.addQueryParam(binding.getKey(), resolveParameterExpression(binding.getValue(), bindingContext));
    }

    for (ParameterBinding binding : requestBindings.getHeaders()) {
      restRequestBuilder.addHeader(binding.getKey(), resolveParameterExpression(binding.getValue(), bindingContext));
    }

    if (requestBindings.getBody() != null) {
      TypedValue<?> payloadTypedValue = resolveRequestPayloadExpression(requestBindings.getBody().getValue(), bindingContext);

      restRequestBuilder.setBody(
                                 toInputStreamTypedValue(payloadTypedValue,
                                                         resolveCharset(empty(), getDefaultResponseMediaType())),
                                 overrides.getStreamingType());
    }

    return restRequestBuilder;
  }

  private DataType getRequestBodyDataType() {
    return toDataType(getRequestBodyMediaType());
  }

  private DataType getResponseBodyDataType() {
    return toDataType(getResponseBodyMediaType());
  }

  private DataType toDataType(String mediaType) {
    return org.mule.runtime.api.metadata.DataType.builder().type(String.class).mediaType(mediaType).build();
  }

  protected String getRequestBodyMediaType() {
    return MediaType.APPLICATION_JSON.toString();
  }

  protected String getResponseBodyMediaType() {
    return MediaType.APPLICATION_JSON.toString();
  }

  protected HttpRequestBinding getRequestBindings() {
    return new HttpRequestBinding();
  }

  protected HttpResponseBinding getResponseBindings() {
    return new HttpResponseBinding();
  }

  // -- Private
  private TypedValue<InputStream> toInputStreamTypedValue(TypedValue<?> typedValue, Charset charset) {
    if (typedValue == null) {
      return null;
    }
    return new TypedValue<>(toInputStream(typedValue, charset), typedValue.getDataType());
  }

  private InputStream toInputStream(TypedValue<?> typedValue, Charset charset) {
    if (typedValue == null) {
      return null;
    }
    InputStream result = null;
    final Object value = typedValue.getValue();
    if (value instanceof String) {
      final String text = (String) value;
      result = new ByteArrayInputStream(text.getBytes(charset));
    } else if (value instanceof CursorStreamProvider) {
      CursorStreamProvider cursorStreamProvider = (CursorStreamProvider) value;
      result = cursorStreamProvider.openCursor();
    }
    return result;
  }

  private BindingContext getResponseBindingContext(Object payload, Map<String, Object> parameterBindings) {
    return BindingContext.builder()
        .addBinding(CONTEXT_KEY_PARAMETERS, getTypedValueOrNull(parameterBindings))
        .addBinding(PAYLOAD_VAR, getTypedValueOrNull(payload))
        .build();
  }

  private BindingContext getRequestBindingContext(RestConnection connection, RestConfiguration config,
                                                  Map<String, Object> parameterBindings) {
    BindingContext.Builder builder = builder()
        .addBinding(CONTEXT_KEY_PARAMETERS, getTypedValueOrNull(parameterBindings))
        .addBinding(CONFIG, getTypedValueOrNull(config.getBindings()))
        .addBinding(CONNECTION, getTypedValueOrNull(connection.getBindings()));

    return builder.build();
  }

  private String resolveParameterExpression(String expression, BindingContext bindingContext) {
    if (!isExpression(expression)) {
      return expression;
    }

    return (String) getExpressionLanguage().evaluate(expression, STRING, bindingContext).getValue();
  }

  private TypedValue<?> resolveRequestPayloadExpression(String expression, BindingContext bindingContext) {
    return resolvePayloadExpression(expression, bindingContext, getRequestBodyDataType());
  }

  private TypedValue<?> resolveResponsePayloadExpression(String expression, BindingContext bindingContext) {
    return resolvePayloadExpression(expression, bindingContext, getResponseBodyDataType());
  }

  private TypedValue<?> resolvePayloadExpression(String expression, BindingContext bindingContext, DataType dataType) {
    if (!isExpression(expression)) {
      return TypedValue.of(expression);
    }

    return getExpressionLanguage().evaluate(expression, dataType, bindingContext);
  }

  protected Result<InputStream, HttpResponseAttributes> postProcessResult(Result<InputStream, HttpResponseAttributes> result,
                                                                          Map<String, Object> parameterBindings) {

    // As we are only transforming the body, we do no transformation if there is no body binding.
    // This way we avoid the relatively heavy operation that executing a DW expression is.
    if (getResponseBindings() == null || getResponseBindings().getBody() == null) {
      return result;
    }

    final Result.Builder<InputStream, HttpResponseAttributes> newResultBuilder = Result.builder();
    final ParameterBinding bodyBinding = getResponseBindings().getBody();

    if (bodyBinding != null) {
      TypedValue<String> payload = consumeStringAndClose(result.getOutput(),
                                                         result.getMediaType().orElse(getDefaultResponseMediaType()),
                                                         resolveCharset(result.getMediaType(), getDefaultResponseMediaType()));

      BindingContext bindingContext = getResponseBindingContext(payload, parameterBindings);

      TypedValue<?> typedValue = resolveResponsePayloadExpression(bodyBinding.getValue(), bindingContext);
      InputStream inputStream = toInputStream(typedValue, resolveCharset(result.getMediaType(), getDefaultResponseMediaType()));

      if (inputStream != null) {
        newResultBuilder.output(inputStream);
      } else {
        throw new MuleRuntimeException(
                                       createStaticMessage("Unexpected '%s' on result of evaluating response binding '%s'",
                                                           typedValue.getValue(),
                                                           bodyBinding));
      }

      newResultBuilder.output(inputStream);
    }

    result.getMediaType().ifPresent(newResultBuilder::mediaType);
    result.getAttributes().ifPresent(newResultBuilder::attributes);
    result.getAttributesMediaType().ifPresent(newResultBuilder::attributesMediaType);
    return newResultBuilder.build();
  }

  protected <T> CompletionCallback<T, HttpResponseAttributes> callbackObjectAttributesAdapter(CompletionCallback<T, Object> callback) {
    return new CompletionCallback<T, HttpResponseAttributes>() {

      @Override
      public void success(Result<T, HttpResponseAttributes> result) {
        callback.success(toObjectResult(result));
      }

      @Override
      public void error(Throwable throwable) {
        callback.error(throwable);
      }
    };
  }

  protected <T> Result<T, Object> toObjectResult(Result<? super T, HttpResponseAttributes> result) {
    Result.Builder<T, Object> builder = Result.builder();
    builder.output((T) result.getOutput());
    result.getMediaType().ifPresent(mediaType -> builder.mediaType(mediaType));
    result.getAttributes().ifPresent(attributes -> builder.attributes(attributes));
    result.getAttributesMediaType().ifPresent(attributesMediaType -> builder.attributesMediaType(attributesMediaType));
    result.getByteLength().ifPresent(byteLength -> builder.length(byteLength));

    return builder.build();
  }

  // End of Operation adapter support

  /**
   * This method return the http response interceptors from the configuration. If needed, this method could be overridden to
   * specify individual interceptors for this operation.
   * 
   * @param config rest configuration to get the default interceptors
   * @return the http response interceptors to be added to the request builder
   */
  protected HttpResponseInterceptor getResponseInterceptors(RestConfiguration config) {
    return config.getResponseInterceptors();
  }

}
