/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.multipart;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;

import com.mulesoft.connectivity.rest.commons.api.multipart.MultipartPayloadBuilder;
import com.mulesoft.connectivity.rest.commons.internal.util.DwUtils;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * A {@link MultipartPayloadBuilder} which uses a DataWeave {@link ExpressionLanguage} to build the body
 *
 * @since 1.0
 */
public class DWMultipartPayloadBuilder implements MultipartPayloadBuilder {

  private final ExpressionLanguage expressionLanguage;
  private final List<Part> parts = new ArrayList<>();
  private String boundary = null;

  public DWMultipartPayloadBuilder(ExpressionLanguage expressionLanguage) {
    this.expressionLanguage = expressionLanguage;
  }

  @Override
  public MultipartPayloadBuilder addPart(String partName, TypedValue<InputStream> content) {
    parts.add(new Part(partName, content));
    return this;
  }

  @Override
  public MultipartPayloadBuilder addFilePart(String partName, String filename, TypedValue<InputStream> content) {
    parts.add(new FilePart(partName, filename, content));
    return this;
  }

  @Override
  public MultipartPayloadBuilder setBoundary(String boundary) {
    this.boundary = boundary;
    return this;
  }

  @Override
  public TypedValue<InputStream> build() {
    return (TypedValue<InputStream>) doBuild(true);
  }

  /**
   * @return a {@link CursorStreamProvider} in order to allow reading more than once the content {@link InputStream}.
   */
  public TypedValue<CursorStreamProvider> asCursorStreamProvider() {
    return (TypedValue<CursorStreamProvider>) doBuild(false);
  }

  private TypedValue doBuild(boolean deferred) {
    StringBuilder builder = new StringBuilder("%dw 2.0\noutput multipart/form-data");
    if (boundary != null) {
      builder.append(" boundary='").append(boundary).append("'");
    }
    if (deferred) {
      if (boundary != null) {
        builder.append(",");
      }
      builder.append(" deferred=true");
    }

    builder.append("\n---\n"
        + "{\n"
        + "parts: {\n");

    BuilderVisitor visitor = new BuilderVisitor(builder);
    BindingContext.Builder bindingContextBuilder = BindingContext.builder();
    parts.forEach(p -> {
      visitor.addPartDwIdentifier(p, DwUtils.getDwIdentifier(p.getName()));
      p.accept(visitor);
      bindingContextBuilder.addBinding(visitor.getPartDwIdentifier(p), p.getContent());
    });

    builder.append("\n}\n}");

    return expressionLanguage.evaluate(builder.toString(), bindingContextBuilder.build());
  }

  private static class BuilderVisitor implements PartVisitor {

    private final StringBuilder builder;
    private boolean partBuilt = false;
    private Map<Part, String> partDwIdentifier;

    public BuilderVisitor(StringBuilder builder) {
      this.builder = builder;
      this.partDwIdentifier = new HashMap<>();
    }

    public void addPartDwIdentifier(Part part, String dwIdentifier) {
      partDwIdentifier.put(part, dwIdentifier);
    }

    public String getPartDwIdentifier(Part part) {
      return partDwIdentifier.get(part);
    }

    @Override
    public void visit(Part part) {
      buildPart(part, emptyList());
    }

    @Override
    public void visit(FilePart part) {
      String header = new StringBuilder("\"Content-Disposition\": {\n")
          .append("name: \"").append(part.getName()).append("\",\n")
          .append("filename: \"").append(part.getFilename()).append("\"\n}")
          .toString();

      buildPart(part, singletonList(header));
    }

    private void buildPart(Part part, List<String> headers) {
      if (partBuilt) {
        builder.append(",\n");
      }

      builder.append("\"").append(part.getName()).append("\": {\n")
          .append("headers: {\n");

      boolean headersAdded = false;
      for (String header : headers) {
        if (headersAdded) {
          builder.append(",\n");
        }
        builder.append(header);
        headersAdded = true;
      }

      if (headersAdded) {
        builder.append(",\n");
      }

      builder.append("\"Content-Type\": \"").append(part.getContent().getDataType().getMediaType().toRfcString())
          .append("\"\n},\ncontent: ").append(getPartDwIdentifier(part)).append("\n}");

      partBuilt = true;
    }
  }
}
