/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.util;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isBlank;
import static java.lang.String.format;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.DataType.BOOLEAN;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_XML;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.TypedValue;

import java.util.Iterator;

/**
 * Utils class for splitting payloads.
 *
 * @since 1.0
 */
public class SplitPayloadUtils {

  /**
   * Splits a payload handling its media type and the needed expressions automatically.
   * 
   * @param expressionLanguage The instance of the expression language to use.
   * @param payload The payload to split.
   * @return An iterator containing an element for each item in the split payload.
   */
  public static Iterator<TypedValue<?>> split(ExpressionLanguage expressionLanguage, TypedValue<?> payload) {
    return split(expressionLanguage, payload, null);
  }

  /**
   * Splits a payload handling its media type and the needed expressions automatically.
   * 
   * @param expressionLanguage The instance of the expression language to use.
   * @param payload The payload to split.
   * @param previousExpression An expression provided by the user that will be pointed to if the split fails to provide better
   *        error context.
   * @return An iterator containing an element for each item in the split payload.
   */
  public static Iterator<TypedValue<?>> split(ExpressionLanguage expressionLanguage, TypedValue<?> payload,
                                              String previousExpression) {
    Iterator<TypedValue<?>> splitResult;
    if (payload.getDataType().getMediaType().matches(APPLICATION_XML)) {
      validatePayload(expressionLanguage, payload, "#[payload is Object]", previousExpression);
      splitResult = expressionLanguage.split("#[payload[0] default []]", buildContext(payload));
    } else {
      validatePayload(expressionLanguage, payload, "#[payload is Array]", previousExpression);
      splitResult = expressionLanguage.split("#[payload]", buildContext(payload));
    }
    return splitResult;
  }

  private static void validatePayload(ExpressionLanguage expressionLanguage,
                                      TypedValue<?> payload,
                                      String expression,
                                      String previousExpression) {
    TypedValue<?> result = expressionLanguage.evaluate(expression, BOOLEAN, buildContext(payload));
    if (result.getValue().equals(false)) {

      final String stringMessage = isBlank(expression)
          ? format("Failed to split payload that is not an array. Expression: '%s'. Result Payload: '%s'.", previousExpression,
                   payload.getValue())
          : format("Failed to split payload that is not an array. Result Payload: '%s'.", payload.getValue());

      throw new MuleRuntimeException(createStaticMessage(stringMessage));
    }
  }

  private static BindingContext buildContext(TypedValue<?> payload) {
    return BindingContext.builder()
        .addBinding("payload", payload).build();
  }

}
