/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.connection.oauth;

import static com.mulesoft.connectivity.rest.commons.api.error.RestError.UNAUTHORIZED;
import static org.mule.runtime.api.el.BindingContext.builder;
import static org.mule.runtime.api.metadata.DataType.BOOLEAN;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionExecutionException;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.i18n.I18nMessageFactory;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.core.api.expression.ExpressionRuntimeException;
import org.mule.runtime.extension.api.connectivity.oauth.AccessTokenExpiredException;
import org.mule.runtime.extension.api.connectivity.oauth.OAuthState;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.auth.HttpAuthentication;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.connection.DefaultRestConnection;
import com.mulesoft.connectivity.rest.commons.api.error.RestError;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.internal.RestConstants;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;

import java.io.InputStream;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

/**
 * Specialization of {@link DefaultRestConnection} for resources protected with OAuth, regardless of the grant type.
 * <p>
 * The main features of this connection are that:
 *
 * <ul>
 * <li>Each instance can only serve one resource owner id. New instances need to be created for each</li>
 * <li>When the service providers responds with a 401 status code, the response {@link CompletableFuture} will be completed with
 * an {@link AccessTokenExpiredException} so that the SDK triggers the refresh token workflow.</li>
 * </ul>
 *
 * @since 1.0
 */
public class OAuthRestConnection extends DefaultRestConnection {

  private final OAuthState oauthState;
  private final String resourceOwnerId;
  private final String refreshTokenCondition;

  /**
   * Creates a new instance
   *
   * @param baseUri the service base uri
   * @param configName the name of the config that owns this connection
   * @param httpClient the client to use
   * @param authentication the authentication mechanism to use, or {@code null}
   * @param defaultQueryParams query params to be automatically added to all requests done through this connection
   * @param defaultHeaders headers to be automatically added to all requests done through this connection
   * @param oauthState the {@link OAuthState} for the current connection
   * @param resourceOwnerId the id of the resource owner for this connection
   * @param expressionLanguage An expression language instance.
   * @param refreshTokenCondition An expression that causes the token to be renewed when return true.
   */
  public OAuthRestConnection(String baseUri,
                             String configName,
                             HttpClient httpClient,
                             HttpAuthentication authentication,
                             MultiMap<String, String> defaultQueryParams,
                             MultiMap<String, String> defaultHeaders,
                             OAuthState oauthState,
                             String resourceOwnerId, ExpressionLanguage expressionLanguage, String refreshTokenCondition) {
    super(baseUri, configName, httpClient, authentication, defaultQueryParams, defaultHeaders, expressionLanguage);
    this.refreshTokenCondition = refreshTokenCondition;

    checkArgument(resourceOwnerId != null, "resourceOwnerId cannot be null");
    this.resourceOwnerId = resourceOwnerId;

    checkArgument(oauthState != null, "oauthState cannot be null");
    this.oauthState = oauthState;
  }

  @Override
  protected HttpRequest buildRequest(RestRequestBuilder requestBuilder) {
    requestBuilder.addHeader(AUTHORIZATION, "Bearer " + oauthState.getAccessToken());
    return super.buildRequest(requestBuilder);
  }

  private BindingContext buildContext(HttpResponse response, MediaType defaultResponseMediaType) {
    final Result<Object, HttpResponseAttributes> result = toResult(response, true, defaultResponseMediaType, null);
    BindingContext.Builder builder =
        builder()
            .addBinding(RestConstants.PAYLOAD_VAR,
                        new TypedValue<>(result.getOutput(),
                                         DataType.builder().mediaType(result.getMediaType().orElse(MediaType.ANY)).build()))
            .addBinding(RestConstants.ATTRIBUTES_VAR, new TypedValue<>(result.getAttributes(), DataType.builder()
                .mediaType(result.getAttributesMediaType().orElse(MediaType.ANY)).build()));
    return builder.build();
  }

  public Optional<String> getRefreshTokenCondition() {
    return Optional.ofNullable(refreshTokenCondition);
  }

  private boolean needsRefresh(HttpResponse response, String refreshTokenCondition, MediaType defaultResponseMediaType) {
    try {
      return (Boolean) getExpressionLanguage()
          .evaluate(refreshTokenCondition, BOOLEAN, buildContext(response, defaultResponseMediaType)).getValue();
    } catch (ExpressionExecutionException | ExpressionRuntimeException e) {
      throw new MuleRuntimeException(I18nMessageFactory
          .createStaticMessage("Failed to execute defined OAuth2 RefreshTokenCondition expression."), e);
    }
  }

  @Override
  protected void handleResponseError(HttpResponse response,
                                     MediaType defaultResponseMediaType,
                                     CompletableFuture<Result<InputStream, HttpResponseAttributes>> future,
                                     StreamingHelper streamingHelper,
                                     RestError error) {
    boolean needsRefresh;

    final String refreshTokenCondition = getRefreshTokenCondition().orElse(null);
    if (refreshTokenCondition != null) {
      response = toRepeatableHttpResponseEntity(response, streamingHelper);
      needsRefresh = needsRefresh(response, refreshTokenCondition, defaultResponseMediaType);
    } else {
      needsRefresh = error == UNAUTHORIZED;
    }

    if (needsRefresh) {
      future.completeExceptionally(new AccessTokenExpiredException(resourceOwnerId));
    } else {
      super.handleResponseError(response, defaultResponseMediaType, future, streamingHelper, error);
    }
  }
}
