/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.interception.expression;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.resolveCharset;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.DataType.BOOLEAN;
import static org.mule.runtime.api.metadata.DataType.fromType;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.server.HttpServerProperties.PRESERVE_HEADER_CASE;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.CompiledExpression;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.Cursor;
import org.mule.runtime.api.streaming.bytes.CursorStream;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.core.internal.streaming.ManagedCursorProvider;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.interception.HttpResponseInterceptor;
import com.mulesoft.connectivity.rest.commons.api.interception.InterceptionHttpRequest;
import com.mulesoft.connectivity.rest.commons.internal.model.http.HttpEntityCursorStreamProviderBased;

import java.io.IOException;
import java.io.InputStream;
import java.util.Optional;
import java.util.OptionalLong;

/**
 * A {@link HttpResponseInterceptor} implementation based on DW expressions. It allows use of DW expressions to define the
 * matching condition and transformations for statusCode, reasonPhrase, headers and body.
 * <p/>
 * {@link BindingContext} provided on each expression has the original {@link HttpResponse} data with the following variables:
 * <ul>
 * <li>statusCode: int</li>
 * <li>reasonPhrase: String</li>
 * <li>headers: MultiMap<String, String></li>
 * <li>body: InputStream</li>
 * </ul>
 */
public class ExpressionHttpResponseInterceptor extends BaseExpressionHttpResponseInterceptor {

  private String matchExpression;

  private String statusCodeExpression;
  private String reasonPhraseExpression;
  private String headersExpression;
  private String bodyExpression;

  private ExpressionHttpResponseInterceptor(String matchExpression,
                                            String statusCodeExpression,
                                            String reasonPhraseExpression,
                                            String headersExpression,
                                            String bodyExpression,
                                            MediaType defaultResponseMediaType) {
    super(defaultResponseMediaType);
    requireNonNull(matchExpression);

    this.matchExpression = matchExpression;

    this.statusCodeExpression = statusCodeExpression;
    this.reasonPhraseExpression = reasonPhraseExpression;
    this.headersExpression = headersExpression;
    this.bodyExpression = bodyExpression;
  }

  @Override
  public boolean match(InterceptionHttpRequest httpRequest, HttpResponse httpResponse, ExpressionLanguage expressionLanguage) {
    return (boolean) evaluate(matchExpression, createBindingContext(httpRequest, httpResponse), BOOLEAN, expressionLanguage)
        .getValue();
  }

  @Override
  public HttpResponse intercept(InterceptionHttpRequest httpRequest, HttpResponse httpResponse,
                                ExpressionLanguage expressionLanguage) {
    int statusCode = (int) getStatusCodeExpression()
        .map(expression -> evaluate(expression, createBindingContext(httpRequest, httpResponse), fromType(Integer.class),
                                    expressionLanguage).getValue())
        .orElseGet(() -> httpResponse.getStatusCode());
    String reasonPhrase = (String) getReasonPhraseExpression()
        .map(expression -> evaluate(expression, createBindingContext(httpRequest, httpResponse), fromType(String.class),
                                    expressionLanguage).getValue())
        .orElseGet(() -> httpResponse.getReasonPhrase());

    MultiMap<String, String> headers =
        (MultiMap<String, String>) getHeadersExpression()
            .map(expression -> evaluate(headersExpression, createBindingContext(httpRequest, httpResponse),
                                        fromType(MultiMap.class),
                                        expressionLanguage).getValue())
            .orElseGet(() -> httpResponse.getHeaders());

    HttpEntity entity = httpResponse.getEntity();
    boolean closeCursorStreamProviderOnHttpResponse = false;
    if (getBodyExpression().isPresent()) {
      String bodyExpression = getBodyExpression().get();
      TypedValue<Object> transformedBody = evaluateBodyExpression(bodyExpression, httpResponse,
                                                                  createBindingContext(httpRequest, httpResponse),
                                                                  expressionLanguage);
      if (transformedBody.getValue() instanceof CursorStreamProvider) {
        // If the expression generates an output it will be a CursorStreamProvider
        CursorStreamProvider cursorStreamProvider = (CursorStreamProvider) transformedBody.getValue();
        entity = new HttpEntityCursorStreamProviderBased(cursorStreamProvider,
                                                         transformedBody.getByteLength());
        if (httpResponse.getEntity() instanceof HttpEntityCursorStreamProviderBased) {
          if (((HttpEntityCursorStreamProviderBased) httpResponse.getEntity())
              .getCursorStreamProvider() != cursorStreamProvider) {
            closeCursorStreamProviderOnHttpResponse = true;
          }
        }
      } else if (transformedBody.getValue() instanceof CursorStream) {
        // Case of an expression that doesn't do a transformation and returns the body binding value (a Cursor)
        entity = new HttpEntityCursorStreamProviderBased(
                                                         (CursorStreamProvider) ((CursorStream) transformedBody.getValue())
                                                             .getProvider(),
                                                         transformedBody.getByteLength());
      } else {
        throw new MuleRuntimeException(
                                       createStaticMessage("HTTP response interceptor expression for body has generated a non recognizable output: '%s'. This is probably a bug.",
                                                           transformedBody.getValue().getClass().getName()));
      }
      headers = updateContentTypeHeader(headers, entity);
    }

    // Last but not least we check if we need to close the CSP for the HttpResponse
    if (closeCursorStreamProviderOnHttpResponse) {
      // Now we can close the previous CSP as the HttpResponse.entity.content has been transformed.
      ((HttpEntityCursorStreamProviderBased) httpResponse.getEntity()).getCursorStreamProvider().close();
    }

    return HttpResponse.builder()
        .statusCode(statusCode)
        .reasonPhrase(reasonPhrase)
        .entity(entity)
        .headers(headers)
        .build();
  }

  private TypedValue<Object> evaluateBodyExpression(String bodyExpression, HttpResponse httpResponse,
                                                    BindingContext bindingContext,
                                                    ExpressionLanguage expressionLanguage) {
    CompiledExpression compile = expressionLanguage.compile(bodyExpression, bindingContext);
    MediaType mediaType = compile.outputType().orElse(getMediaType(httpResponse));
    return (TypedValue<Object>) expressionLanguage
        .evaluate(bodyExpression, DataType.builder()
            .mediaType(mediaType)
            .charset(resolveCharset(empty(), mediaType))
            .build(),
                  bindingContext);
  }

  private MultiMap<String, String> updateContentTypeHeader(MultiMap<String, String> headers, HttpEntity entity) {
    if (!headers.containsKey(CONTENT_LENGTH) && !headers.containsKey(CONTENT_LENGTH.toLowerCase())) {
      return headers;
    }
    MultiMap.StringMultiMap modifiedHeaders = new MultiMap.StringMultiMap();
    for (String key : headers.keySet()) {
      if (!key.equalsIgnoreCase(CONTENT_LENGTH)) {
        modifiedHeaders.put(key, headers.getAll(key));
      }
    }
    try {
      modifiedHeaders.put(PRESERVE_HEADER_CASE ? CONTENT_LENGTH : CONTENT_LENGTH.toLowerCase(),
                          String.valueOf(entity.getBytes().length));
      return modifiedHeaders;
    } catch (IOException e) {
      throw new MuleRuntimeException(createStaticMessage(
                                                         "There was an error while trying to resolve the content length for the transformed body response"),
                                     e);
    }
  }

  private Optional<String> getBodyExpression() {
    return ofNullable(bodyExpression);
  }

  private Optional<String> getHeadersExpression() {
    return ofNullable(headersExpression);
  }

  private Optional<String> getStatusCodeExpression() {
    return ofNullable(statusCodeExpression);
  }

  private Optional<String> getReasonPhraseExpression() {
    return ofNullable(reasonPhraseExpression);
  }

  /**
   * @return a {@link ExpressionHttpResponseInterceptorBuilder} to create a {@link ExpressionHttpResponseInterceptor}.
   */
  public static ExpressionHttpResponseInterceptorBuilder builder() {
    return new ExpressionHttpResponseInterceptorBuilder();
  }

  /**
   * Builder for a {@link ExpressionHttpResponseInterceptor}.
   */
  public static class ExpressionHttpResponseInterceptorBuilder {

    private String matchExpression;

    protected String statusCodeExpression;
    protected String reasonPhraseExpression;
    protected String headersExpression;
    protected String bodyExpression;

    protected MediaType defaultResponseMediaType;

    public ExpressionHttpResponseInterceptorBuilder matchExpression(String matchExpression) {
      requireNonNull(matchExpression);
      this.matchExpression = matchExpression;
      return this;
    }

    /**
     * Sets the expression for resolving the status code.
     *
     * @param statusCodeExpression a DW expression that resolves the status code.
     * @return this
     */
    public ExpressionHttpResponseInterceptorBuilder statusCodeExpression(String statusCodeExpression) {
      requireNonNull(statusCodeExpression);
      this.statusCodeExpression = statusCodeExpression;
      return this;
    }

    /**
     * Sets the expression for resolving the reason phrase.
     *
     * @param reasonPhraseExpression a DW expression that resolves the reason phrase.
     * @return this
     */
    public ExpressionHttpResponseInterceptorBuilder reasonPhraseExpression(String reasonPhraseExpression) {
      requireNonNull(reasonPhraseExpression);
      this.reasonPhraseExpression = reasonPhraseExpression;
      return this;
    }

    public ExpressionHttpResponseInterceptorBuilder headersExpression(String headersExpression) {
      requireNonNull(headersExpression);
      this.headersExpression = headersExpression;
      return this;
    }

    /**
     * Sets the expression for resolving the body.
     *
     * @param bodyExpression a DW expression that resolves the body.
     * @return this
     */
    public ExpressionHttpResponseInterceptorBuilder bodyExpression(String bodyExpression) {
      requireNonNull(bodyExpression);
      this.bodyExpression = bodyExpression;
      return this;
    }

    /**
     * Defines the default response media type for the {@link HttpResponse} that is intercepted by the interceptor defined by this
     * builder.
     *
     * @param defaultResponseMediaType default response media type for the {@link HttpResponse} that is intercepted by the
     *        interceptor defined by this builder.
     * @return this
     */
    public ExpressionHttpResponseInterceptorBuilder defaultResponseMediaType(MediaType defaultResponseMediaType) {
      requireNonNull(defaultResponseMediaType);
      this.defaultResponseMediaType = defaultResponseMediaType;
      return this;
    }

    /**
     * @return a instance of {@link ExpressionHttpResponseInterceptor}.
     */
    public ExpressionHttpResponseInterceptor build() {
      return new ExpressionHttpResponseInterceptor(matchExpression, statusCodeExpression, reasonPhraseExpression,
                                                   headersExpression,
                                                   bodyExpression, defaultResponseMediaType);
    }

  }

}
