/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.metadatamodel.handler;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.commons.internal.util.StringCaseUtils.titleCase;
import static java.util.Comparator.comparing;

import com.mulesoft.connectivity.rest.commons.internal.util.StringCaseUtils;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;

import java.util.AbstractMap.SimpleEntry;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.everit.json.schema.ObjectSchema;
import org.everit.json.schema.Schema;

/**
 * This class is a copy of {@link org.mule.metadata.json.api.handler.ObjectHandler} that supports the custom logic required to
 * propagate the composer metadata fields from the Json Schema to the extension model.
 *
 * This class does not respect the actual standard of how the extension model works, and can be understood by composer by
 * convention.
 *
 * This should be removed once metadata model has proper support for custom annotations.
 */
public class RestObjectHandler implements SchemaHandler {

  // Semi standard way of declaring an order for the properties of a Json Schema
  // https://github.com/json-schema/json-schema/issues/119
  // https://github.com/quicktype/quicktype/pull/1340
  public static final String PROPERTY_ORDER = "propertyOrder";
  public static final String CUSTOM_JSON_PROPERTY_PREFIX = "x-";

  @Override
  public boolean handles(Schema schema) {
    return schema instanceof ObjectSchema;
  }

  @Override
  public TypeBuilder<?> handle(Schema schema, BaseTypeBuilder root, RestHandlerManager handlerManager,
                               ParsingContext parsingContext) {

    final ObjectSchema objectSchema = (ObjectSchema) schema;
    final ObjectTypeBuilder objectMetadataBuilder = root.objectType().ordered(true);
    parsingContext.registerBuilder(objectSchema, objectMetadataBuilder);

    final String id = objectSchema.getId();
    if (isNotBlank(id)) {
      objectMetadataBuilder.id(id);
    }

    getObjectLabel(objectSchema).ifPresent(objectMetadataBuilder::label);

    final List<Map.Entry<String, Schema>> orderedProperties = getOrderedProperties(objectSchema);
    final List<String> requiredProperties = objectSchema.getRequiredProperties();

    // Add named properties
    for (Map.Entry<String, Schema> property : orderedProperties) {
      final ObjectFieldTypeBuilder field = objectMetadataBuilder.addField().key(property.getKey());
      processSchemaProperty(handlerManager, parsingContext, requiredProperties, property, field);
    }

    final Map<Pattern, Schema> patternProperties = objectSchema.getPatternProperties();
    // Sort them by pattern
    final Collection<Map.Entry<Pattern, Schema>> entries =
        patternProperties.entrySet().stream()
            .sorted(comparing(o -> o.getKey().toString()))
            .collect(Collectors.toList());

    for (Map.Entry<Pattern, Schema> patternSchemaEntry : entries) {
      final ObjectFieldTypeBuilder field = objectMetadataBuilder.addField();
      field.key(patternSchemaEntry.getKey());
      final Schema value = patternSchemaEntry.getValue();
      field.value(handlerManager.handle(value, parsingContext));
    }
    if (objectSchema.permitsAdditionalProperties()) {
      objectMetadataBuilder.openWith(handlerManager.handle(objectSchema.getSchemaOfAdditionalProperties(), parsingContext));
    }

    return objectMetadataBuilder;
  }

  protected void processSchemaProperty(RestHandlerManager handlerManager, ParsingContext parsingContext,
                                       List<String> requiredProperties, Map.Entry<String, Schema> property,
                                       ObjectFieldTypeBuilder field) {
    field.required(requiredProperties.contains(property.getKey()));
    final Schema value = property.getValue();
    field.value(handlerManager.handle(value, parsingContext));
    optional(() -> getFieldLabel(property.getKey(), property.getValue()),
             () -> inferFieldLabel(property.getKey(), property.getValue())).ifPresent(field::label);
    optional(() -> getFieldDescription(property.getKey(), property.getValue()),
             () -> inferFieldDescription(property.getKey(), property.getValue())).ifPresent(field::description);
  }

  public static <T> Optional<T> optional(Supplier<Optional<T>> optional1Supplier, Supplier<Optional<T>> optional2Supplier) {
    final Optional<T> optional = optional1Supplier.get();
    return optional.isPresent() ? optional : optional2Supplier.get();
  }

  public static Optional<String> nonBlank(String text) {
    return Optional.ofNullable(isNotBlank(text) ? text : null);
  }

  protected Optional<String> getObjectLabel(ObjectSchema objectSchema) {
    return nonBlank(objectSchema.getTitle());
  }

  protected Optional<String> getFieldLabel(String id, Schema property) {
    return nonBlank(property.getTitle());
  }

  protected Optional<String> inferFieldLabel(String id, Schema property) {
    return nonBlank(titleCase(id));
  }

  protected Optional<String> getFieldDescription(String id, Schema property) {
    return nonBlank(property.getDescription());
  }

  protected Optional<String> inferFieldDescription(String id, Schema property) {
    return optional(() -> nonBlank(property.getTitle()), () -> nonBlank(id)).map(StringCaseUtils::sentenceCase);
  }

  private List<Map.Entry<String, Schema>> getOrderedProperties(ObjectSchema objectSchema) {
    ArrayList<?> propertyOrderArray = getPropertyOrder(objectSchema);
    Map<String, Schema> propertySchemas = new HashMap<>(objectSchema.getPropertySchemas());

    if (propertyOrderArray == null) {
      return new ArrayList<>(propertySchemas.entrySet());
    }

    List<Map.Entry<String, Schema>> result = new ArrayList<>();

    for (Object propertyOrderItem : propertyOrderArray) {
      if (propertyOrderItem instanceof String) {
        String key = (String) propertyOrderItem;
        Schema schema = propertySchemas.get(key);
        if (schema != null) {
          propertySchemas.remove(key);
          result.add(new SimpleEntry<>(key, schema));
        }
      }
    }

    result.addAll(propertySchemas.entrySet());

    return result;
  }

  private ArrayList<?> getPropertyOrder(ObjectSchema objectSchema) {
    // We need to handle both propertyOrder and x-propertyOrder as this is not part of the standard and some validators won't
    // allow a custom property that does not start with `x-`.
    ArrayList<?> propertyOrder = getPropertyOrder(objectSchema, PROPERTY_ORDER);
    if (propertyOrder == null) {
      propertyOrder = getPropertyOrder(objectSchema, CUSTOM_JSON_PROPERTY_PREFIX + PROPERTY_ORDER);
    }
    return propertyOrder;
  }

  private ArrayList<?> getPropertyOrder(ObjectSchema objectSchema, String propertyName) {
    if (objectSchema.getUnprocessedProperties().containsKey(propertyName)) {
      Object propertyOrderObject = objectSchema.getUnprocessedProperties().get(propertyName);
      if (propertyOrderObject instanceof ArrayList) {
        return (ArrayList<?>) propertyOrderObject;
      }
    }
    return null;
  }
}
