/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.model.common;

import static org.mule.runtime.api.el.BindingContext.builder;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.CompiledExpression;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.DataTypeBuilder;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;

import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.internal.model.resolvers.ResolverDeclaration;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * The default implementation of an evaluation context.
 *
 * @since 1.0
 */
@SuppressWarnings("rawtypes")
public class SimpleEvaluationContext implements EvaluationContext {

  private final EvaluationContext globalEvaluationContext;

  private final Map<String, Object> properties;
  private final Map<String, Object> variables;
  private final Map<String, ResolverDeclaration> declarations;

  public SimpleEvaluationContext() {
    this(null);
  }

  public SimpleEvaluationContext(EvaluationContext globalEvaluationContext) {
    this.globalEvaluationContext = globalEvaluationContext;
    variables = new HashMap<>();
    properties = new HashMap<>();
    declarations = new HashMap<>();
  }

  @Override
  public Map<String, Object> getVariables() {
    return variables;
  }

  public Optional<EvaluationContext> getGlobalEvaluationContext() {
    return Optional.ofNullable(globalEvaluationContext);
  }

  @Override
  public Object resolve(String expression) {
    ExpressionLanguage expressionLanguage = (ExpressionLanguage) resolveProperty(PROPERTY_EXPRESSION_LANGUAGE);
    BindingContext.Builder bindingContextBuilder = builder();
    getVariables().forEach((s, o) -> bindingContextBuilder.addBinding(s, TypedValue.of(o)));

    BindingContext buildingContext = bindingContextBuilder.build();
    DataType dt = inferDataTypeOrGetDefault(expression, expressionLanguage, buildingContext);
    final TypedValue<?> result = expressionLanguage.evaluate(expression, dt, buildingContext);
    if (result == null) {
      throw new RuntimeException(String.format("Unresolvable expression '%s'", expression));
    } else {
      return result;
    }
  }

  private DataType inferDataTypeOrGetDefault(String expression, ExpressionLanguage expressionLanguage,
                                             BindingContext buildingContext) {
    CompiledExpression compile = expressionLanguage.compile(expression, buildingContext);
    Optional<MediaType> mediaType = compile.outputType();

    DataTypeBuilder dataTypeBuilder = DataType.builder();
    if (mediaType.isPresent()) {
      // scenario when script contains directive like "output application/xml --- <expression>"
      dataTypeBuilder.mediaType(mediaType.get());
    } else {
      // default scenario
      RestConfiguration restConfiguration = (RestConfiguration) resolveProperty(PROPERTY_CONFIGURATION);
      dataTypeBuilder.mediaType(restConfiguration.getDefaultExpressionOutputMediaType());
    }
    return dataTypeBuilder.build();
  }

  @Override
  public Object resolveProperty(String property) {
    return Optional.ofNullable(properties.get(property))
        .orElseGet(() -> getGlobalEvaluationContext()
            .map(evaluationContext -> evaluationContext.resolveProperty(property)).orElse(null));
  }

  @Override
  public void defineProperty(String property, Object value) {
    properties.put(property, value);
  }

  @Override
  public void define(String variable, Object value) {
    variables.put(variable, value);
  }

  @Override
  public ResolverDeclaration resolveDeclaration(String declarationName) {
    final ResolverDeclaration result =
        Optional.ofNullable(declarations.get(declarationName))
            .orElseGet(() -> getGlobalEvaluationContext()
                .map(evaluationContext -> evaluationContext.resolveDeclaration(declarationName)).orElse(null));

    if (result == null) {
      throw new RuntimeException(String.format("Unresolvable declaration '%s'", declarationName));
    } else {
      return result;
    }
  }

  @Override
  public void declare(ResolverDeclaration resolverDeclaration) {
    declarations.put(resolverDeclaration.getName(), resolverDeclaration);
  }
}
