/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.util;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.consumeStringAndClose;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.resolveCharset;
import static java.lang.Thread.currentThread;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.core.internal.streaming.bytes.ByteArrayCursorStreamProvider;
import org.mule.runtime.extension.api.runtime.operation.Result;

import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;

import java.io.InputStream;
import java.util.concurrent.ExecutionException;

/**
 * Utils class that provides easy ways of doing requests and handling its stream responses.
 *
 * @since 1.0
 */
public class RequestStreamingUtils {

  /**
   * Does a request as specified in the provided request builder using the connection, config and media type provided.
   * 
   * @param connection The connection that will make the request.
   * @param config The config from where settings will be extracted for the request.
   * @param requestBuilder The request builder for the request this method should do.
   * @param defaultResponseMediaType The default response media type, will be overwritten by the actual response media type if
   *        provided in the server response.
   * @return A result containing an String typed value obtained by consuming the server response and the HttpResponseAttributes
   *         from the server response.
   */
  public static Result<TypedValue<String>, HttpResponseAttributes> doRequestAndConsumeString(RestConnection connection,
                                                                                             RestConfiguration config,
                                                                                             RestRequestBuilder requestBuilder,
                                                                                             MediaType defaultResponseMediaType) {

    MediaType mediaType = resolveDefaultResponseMediaType(config, defaultResponseMediaType);
    Result<InputStream, HttpResponseAttributes> result = doRequest(connection, config, requestBuilder, mediaType);

    TypedValue<String> rawContent = consumeStringAndClose(result.getOutput(),
                                                          result.getMediaType().orElse(mediaType),
                                                          resolveCharset(result.getMediaType(), mediaType));

    return Result.<TypedValue<String>, HttpResponseAttributes>builder()
        .output(rawContent)
        .mediaType(rawContent.getDataType().getMediaType())
        .attributes(result.getAttributes().orElse(null))
        .attributesMediaType(result.getAttributesMediaType().orElse(null))
        .build();
  }

  public static Result<InputStream, HttpResponseAttributes> doRequest(RestConnection connection,
                                                                      RestConfiguration config,
                                                                      RestRequestBuilder requestBuilder,
                                                                      MediaType mediaType) {
    Result<InputStream, HttpResponseAttributes> result;


    try {
      result =
          connection
              .request(requestBuilder, getConfigResponseTimeout(config), mediaType, null)
              .get();
    } catch (ExecutionException e) {
      Throwable cause = e.getCause();
      if (cause instanceof RuntimeException) {
        throw (RuntimeException) cause;
      }
      throw new MuleRuntimeException(createStaticMessage(cause.getMessage()), cause);
    } catch (InterruptedException e) {
      currentThread().interrupt();
      throw new MuleRuntimeException(e);
    }

    return result;
  }

  private static int getConfigResponseTimeout(RestConfiguration config) {
    return (int) config.getResponseTimeoutUnit().toMillis(config.getResponseTimeout());
  }

  private static MediaType resolveDefaultResponseMediaType(RestConfiguration config, MediaType defaultResponseMediaType) {
    MediaType mediaType = defaultResponseMediaType;
    if (!mediaType.getCharset().isPresent()) {
      mediaType = mediaType.withCharset(config.getCharset());
    }

    return mediaType;
  }

  /**
   * Takes the result that a DW split would generate as input and generates a CursorStreamProvider that allows handling it easily.
   * 
   * @param typedValue The result from a DW split
   * @return An easy to handle CursorStreamProvider.
   */
  public static TypedValue<CursorStreamProvider> getCursorStreamProviderValueFromSplitResult(TypedValue<?> typedValue) {
    if (typedValue.getValue() instanceof CursorStreamProvider) {
      return new TypedValue<>((CursorStreamProvider) typedValue.getValue(), typedValue.getDataType());
    } else if (typedValue.getValue() instanceof String) {
      return new TypedValue<>(new ByteArrayCursorStreamProvider(((String) typedValue.getValue()).getBytes()),
                              typedValue.getDataType());
    } else {
      throw new IllegalArgumentException("Could not create CursorStreamProvider for type: " + typedValue.getValue().getClass());
    }
  }
}
