/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.util;

import static com.mulesoft.connectivity.rest.commons.api.configuration.StreamingType.ALWAYS;
import static com.mulesoft.connectivity.rest.commons.api.configuration.StreamingType.NEVER;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder.ParameterArrayFormat.MULTIMAP;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.stringValue;
import static java.lang.String.CASE_INSENSITIVE_ORDER;
import static java.lang.String.format;
import static java.util.Collections.unmodifiableMap;
import static java.util.Objects.requireNonNull;
import static org.mule.runtime.core.api.util.IOUtils.toByteArray;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.runtime.http.api.HttpHeaders.Names.TRANSFER_ENCODING;
import static org.mule.runtime.http.api.HttpHeaders.Values.CHUNKED;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.HttpConstants.Method;
import org.mule.runtime.http.api.domain.entity.ByteArrayHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import com.mulesoft.connectivity.rest.commons.api.configuration.StreamingType;
import com.mulesoft.connectivity.rest.commons.api.interception.HttpResponseInterceptor;
import com.mulesoft.connectivity.rest.commons.api.operation.RequestParameters;
import com.mulesoft.connectivity.rest.commons.api.operation.queryparam.CommaRequestParameterFormatter;
import com.mulesoft.connectivity.rest.commons.api.operation.queryparam.MultimapRequestParameterFormatter;
import com.mulesoft.connectivity.rest.commons.api.operation.queryparam.RequestParameterFormatter;

import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

import org.slf4j.Logger;


/***
 * Builder of {@link HttpRequest}s. Handles ignoring null queryParams and Headers by default. Provides the ability of setting a
 * format for query parameter arrays. Handles request streaming mode (Content-Length and Transfer-Encoding headers).
 *
 * @since 1.0
 */
public class RestRequestBuilder {

  private static final Logger LOGGER = getLogger(RestRequestBuilder.class);
  public static final String PATH_PARAMETERS_NOT_USED_ERROR_TEMPLATE =
      "The request was not correctly built. Required URI parameters were not set. [%s]";
  public static final String NULL_URI_PARAMETER_ERROR_TEMPLATE =
      "URI Parameters are mandatory and its value cannot be null. Trying to assign a null value to the '%s' URI parameter.";

  private final String baseUri;
  private String path;
  private final Method method;
  private String fullUri;
  private final Map<String, Object> headers = new TreeMap<>(CASE_INSENSITIVE_ORDER);
  private final Map<String, Object> queryParams = new HashMap<>();
  private final HashMap<String, Object> uriParams = new HashMap<>();
  private TypedValue<InputStream> body = null;
  private StreamingType streamingType = StreamingType.AUTO;
  private ParameterArrayFormat queryParamFormat = MULTIMAP;
  private HttpResponseInterceptor httpResponseInterceptor;
  private final RequestParameters customOperationParameters;

  /**
   * Creates a preconfigured {@link RestRequestBuilder} per the given parameters.
   * <p>
   * This creates a new {@link RestRequestBuilder} and initializes its URI and method.
   * <p>
   * The returned builder can still to add/change additional information.
   *
   * @param baseUri the base uri to be used.
   * @param path the path of the resource to request, relative to the base URI.
   * @param method the request method.
   */
  public RestRequestBuilder(String baseUri, String path, Method method) {
    this(baseUri, path, method, null);
  }

  /**
   * Creates a preconfigured {@link RestRequestBuilder} per the given parameters.
   * <p>
   * This creates a new {@link RestRequestBuilder} and initializes its URI, method, custom query params and custom headers.
   * <p>
   * The returned builder can still to add/change additional information.
   *
   * @param baseUri the base uri to be used.
   * @param path the path of the resource to request, relative to the base URI.
   * @param method the request method.
   * @param customParameters a list of custom request parameters that will always be sent in the final request.
   */
  public RestRequestBuilder(String baseUri, String path, Method method, RequestParameters customParameters) {
    this.baseUri = baseUri;
    this.path = path;
    this.method = method;
    this.customOperationParameters = customParameters;
  }

  /***
   * Returns the full uri for this request. (Base Uri + Path)
   */
  public String getUri() {
    return buildRequestUri();
  }

  /***
   * Returns the HTTP verb to be used in this request
   */
  public String getMethod() {
    return method.name();
  }

  /***
   * Adds an uri parameter to the request builder. It will only be added if it's value is not null. URI parameters will be
   * replaced in the path where a placeholder matches its key. This method will use a best effort approach to convert the Object
   * to a valid String representation. If a specific format is needed, send the value as String.
   *
   * @param placeholderKey The header name.
   * @param value The header value. Will be converted to string.
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder addUriParam(String placeholderKey, Object value) {
    if (value != null) {
      uriParams.put(placeholderKey, value);
    } else {
      throw new IllegalArgumentException(
                                         format(NULL_URI_PARAMETER_ERROR_TEMPLATE, placeholderKey));
    }
    return this;
  }

  /***
   * Adds a header to the request builder. It will only be added if it's value is not null. This method will use a best effort
   * approach to convert the Object to a valid String representation. If a specific format is needed, send the value as String.
   *
   * @param key The header name.
   * @param value The header value. Will be converted to string.
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder addHeader(String key, Object value) {
    if (value != null) {
      headers.put(key, value);
    }
    return this;
  }

  /***
   * Adds a query parameter to the request builder. It will only be added if it's value is not null. This method will use a best
   * effort approach to convert the Object to a valid String representation. If a specific format is needed, send the value as
   * String.
   *
   * @param key The query parameter name.
   * @param value The query parameter value. Will be converted to string.
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder addQueryParam(String key, Object value) {
    if (value != null) {
      queryParams.put(key, value);
    }
    return this;
  }

  /***
   * Clears the uri parameters configured for this request.
   */
  public void clearUriParams() {
    uriParams.clear();
  }

  /***
   * Clears the query parameters configured for this request.
   */
  public void clearQueryParams() {
    queryParams.clear();
  }

  /***
   * Clears the header parameters configured for this request.
   */
  public void clearHeaders() {
    headers.clear();
  }

  /***
   * Returns an immutable map containing the uri parameters configured for this request.
   */
  public Map<String, String> getUriParams() {
    return unmodifiableMap(MULTIMAP.getFormatter().format(uriParams));
  }

  /***
   * Returns an immutable map containing the query parameters configured for this request.
   */
  public MultiMap<String, String> getQueryParams() {
    return queryParamFormat.getFormatter().format(queryParams).toImmutableMultiMap();
  }

  /***
   * Returns an immutable map containing the headers configured for this request.
   */
  public MultiMap<String, String> getHeaders() {
    return MULTIMAP.getFormatter().format(headers, false).toImmutableMultiMap();
  }

  /***
   * COMMA: key=value1,value2,value3. MULTIMAP: key=value1&key=value2&key=value3.
   */
  public enum ParameterArrayFormat {

    MULTIMAP(new MultimapRequestParameterFormatter()), COMMA(new CommaRequestParameterFormatter());

    private final RequestParameterFormatter formatter;

    ParameterArrayFormat(RequestParameterFormatter formatter) {
      this.formatter = formatter;
    }

    public RequestParameterFormatter getFormatter() {
      return formatter;
    }
  }

  /***
   * Defines the format in which the query parameter arrays must be handled by this request.
   * 
   * @param queryParamFormat The query parameter array format
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder setQueryParamFormat(ParameterArrayFormat queryParamFormat) {
    this.queryParamFormat = queryParamFormat;
    return this;
  }

  /***
   * Defines the body and content streaming for this request.
   * 
   * @param body A {@link TypedValue} that contains the body to be sent in this request. Nullable.
   * @param streamingType The streaming type that must be used for this request's content.
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder setBody(TypedValue<InputStream> body, StreamingType streamingType) {
    this.body = body;
    this.streamingType = streamingType;
    return this;
  }

  /***
   * Defines a post process {@link HttpResponseInterceptor} to intercept and modify server's response before this is mapped to a
   * {@link org.mule.runtime.extension.api.runtime.operation.Result}.
   *
   * @param httpResponseInterceptor A {@link HttpResponseInterceptor} to intercept server's response.
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder responseInterceptor(HttpResponseInterceptor httpResponseInterceptor) {
    requireNonNull(httpResponseInterceptor);
    this.httpResponseInterceptor = httpResponseInterceptor;
    return this;
  }

  /**
   * @return a post process {@link HttpResponseInterceptor} to intercept and modify server's response before this is mapped to a
   *         {@link org.mule.runtime.extension.api.runtime.operation.Result}.
   */
  public HttpResponseInterceptor getResponseInterceptor() {
    return httpResponseInterceptor;
  }

  /**
   * Generates an {@link RestRequestBuilder} that reflects the request body. Sets the media-type and content-length headers for
   * this builder based on the entity and the request's streaming mode.
   *
   * @param body the request body (nullable)
   * @param streamingType the streaming mode for this request
   * @return an http entity to use in the request body. Null if the body must be empty.
   */
  private HttpEntity getStreamingConfiguredHttpEntity(TypedValue<InputStream> body, StreamingType streamingType) {
    if (body != null) {
      byte[] bytes = null;

      if (streamingType.equals(ALWAYS)) {
        headers.remove(CONTENT_LENGTH);
        headers.remove(TRANSFER_ENCODING);
        addHeader(TRANSFER_ENCODING, CHUNKED);
      } else if (streamingType.equals(NEVER)) {
        bytes = setNeverStreamingContentLength(body);
      } else {
        setAutoContentLengthHeader(body);
      }

      inferContentTypeFromBody(body);

      if (body.getValue() != null) {
        if (bytes != null) {
          return new ByteArrayHttpEntity(bytes);
        } else {
          return new InputStreamHttpEntity(body.getValue());
        }
      }
    }
    return null;
  }

  private void inferContentTypeFromBody(TypedValue<InputStream> body) {
    if (!headers.containsKey(CONTENT_TYPE)) {
      MediaType mediaType = body.getDataType().getMediaType();
      if (mediaType != null && !mediaType.getPrimaryType().equals("*")) {
        addHeader(CONTENT_TYPE, mediaType.toRfcString());
      }
    }
  }

  /**
   * Sets the content length header value for this requests, reading the body to get it if necessary. If the body has been read
   * this method returns it's bytes.
   * 
   * @param body The request body
   * @return A byte[] containing the body if the input stream has been read.
   */
  private byte[] setNeverStreamingContentLength(TypedValue<InputStream> body) {
    String customLength = stringValue(headers.get(CONTENT_LENGTH));

    byte[] bytes = null;

    if (customLength == null) {
      if (body.getByteLength().isPresent()) {
        addHeader(CONTENT_LENGTH, body.getByteLength().getAsLong());
      } else if (body.getValue() != null) {
        bytes = toByteArray(body.getValue());
        addHeader(CONTENT_LENGTH, bytes.length);
      }
    }

    headers.remove(TRANSFER_ENCODING);

    return bytes;
  }

  private void setAutoContentLengthHeader(TypedValue<InputStream> body) {
    MultiMap<String, String> headers = getHeaders();

    String customLength = stringValue(headers.get(CONTENT_LENGTH));
    boolean isChunked = CHUNKED.equals(headers.get(TRANSFER_ENCODING));

    if (body.getByteLength().isPresent()) {
      boolean addHeader = true;
      String length = stringValue(body.getByteLength().getAsLong());
      if (customLength != null) {
        LOGGER.warn("Invoking URI {} with body of known length {}. However, a {} header with value {} was manually specified. "
            + "Will proceed with the custom value.",
                    getUri(), length, CONTENT_LENGTH, customLength);

        addHeader = false;
      }

      if (isChunked) {
        LOGGER.debug("Invoking URI {} with a manually set {}: {} header, even though body is of known length {}. "
            + "Skipping automatic addition of {} header", getUri(), TRANSFER_ENCODING, CHUNKED, length, CONTENT_LENGTH);
        addHeader = false;
      }

      if (addHeader) {
        addHeader(CONTENT_LENGTH, length);
      }

    } else if (customLength == null && !isChunked) {
      addHeader(TRANSFER_ENCODING, CHUNKED);
    }
  }

  /***
   * @return The configured HttpRequest
   */
  public HttpRequest build() {
    // Add all custom operation parameters to the request.
    // Custom parameters override the ones set in other parts of the builder.
    if (customOperationParameters != null) {
      MultiMap<String, String> customQueryParams = customOperationParameters.getCustomQueryParams();
      for (String key : customQueryParams.keySet()) {
        addQueryParam(key, customQueryParams.getAll(key));
      }
      MultiMap<String, String> customHeaders = customOperationParameters.getCustomHeaders();
      for (String key : customHeaders.keySet()) {
        addHeader(key, customHeaders.getAll(key));
      }
    }

    // Build the entity for this request.
    // This also automatically sets some headers like content-type, content-length and transfer-encoding if needed.
    HttpEntity httpEntity = getStreamingConfiguredHttpEntity(body, streamingType);

    // Build and validate the uri for the request.
    String requestUri = getUri();
    validateUri(requestUri);

    HttpRequestBuilder builder = HttpRequest.builder(true)
        .uri(requestUri)
        .method(method)
        .queryParams(getQueryParams())
        .headers(getHeaders());

    if (httpEntity != null) {
      builder.entity(httpEntity);
    }

    return builder.build();
  }

  private void validateUri(String requestUri) {
    if (requestUri.contains("{")) {
      throw new IllegalArgumentException(
                                         format(PATH_PARAMETERS_NOT_USED_ERROR_TEMPLATE, requestUri));
    }
  }

  /**
   * Sets the path section for the request URI
   *
   * @param path the path of the request.
   */
  public void setPath(String path) {
    this.path = path;
  }

  /**
   * Sets a full request URI that overrides the configured baseUri and path.
   * 
   * @param fullUri to be set as the request URI
   */
  public void setFullUri(String fullUri) {
    this.fullUri = fullUri;
  }

  private String buildRequestUri() {
    if (isNotBlank(fullUri)) {
      return fullUri;
    }

    String localPath = path;
    String localBaseUri = baseUri;

    boolean pathStartsWithSlash = localPath != null && localPath.startsWith("/");
    boolean baseEndsInSlash = localBaseUri.endsWith("/");

    if (pathStartsWithSlash && baseEndsInSlash) {
      localPath = localPath.substring(1);
    } else if (!pathStartsWithSlash && !baseEndsInSlash) {
      localBaseUri += '/';
    }

    localPath = applyUriParameters(localPath);

    if (localPath == null) {
      return localBaseUri;
    }
    return localBaseUri + localPath;
  }

  private String applyUriParameters(String path) {
    Map<String, String> effectiveUriParams = getUriParams();

    for (String key : effectiveUriParams.keySet()) {
      path = path.replace(format("{%s}", key), effectiveUriParams.get(key));
    }

    return path;
  }
}
