/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.configuration;

import static java.util.Collections.emptyMap;
import static java.util.concurrent.TimeUnit.SECONDS;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.extension.api.annotation.param.display.Placement.ADVANCED_TAB;

import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.DefaultEncoding;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.annotation.values.OfValues;

import com.mulesoft.connectivity.rest.commons.api.datasense.valueprovider.StreamingTypeValueProvider;
import com.mulesoft.connectivity.rest.commons.api.interception.descriptor.HttpResponseInterceptorDescriptor;

import java.nio.charset.Charset;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Base class for configuration objects
 *
 * @since 1.0
 */
public abstract class RestConfiguration implements Initialisable {

  @DefaultEncoding
  private String defaultEncoding;
  private Charset charset;

  /**
   * The timeout for request to the remote service. This value is qualified by the {@link #responseTimeoutUnit}
   */
  @Parameter
  @Optional(defaultValue = "60")
  @Placement(tab = ADVANCED_TAB)
  @Expression(NOT_SUPPORTED)
  @Summary("The timeout for request to the remote service.")
  private Integer responseTimeout;

  /**
   * A time unit which qualifies the {@link #responseTimeout}
   */
  @Parameter
  @Optional(defaultValue = "SECONDS")
  @Placement(tab = ADVANCED_TAB)
  @Expression(NOT_SUPPORTED)
  @Summary("A time unit which qualifies the Response Timeout}")
  private TimeUnit responseTimeoutUnit = SECONDS;

  /**
   * Defines if the request should be sent using streaming or not. Setting the value to AUTO will automatically define the best
   * strategy based on the request content. As streaming is done the request will be sent user Transfer-Encoding: chunked.
   */
  @Parameter
  @Placement(tab = ADVANCED_TAB)
  @Optional(defaultValue = "AUTO")
  @Expression(NOT_SUPPORTED)
  @Summary("Defines if the request should be sent using streaming. "
      + "Setting the value to AUTO will automatically define the best strategy based on the request content.")
  @OfValues(StreamingTypeValueProvider.class)
  private String streamingType;
  private StreamingType streamingTypeEnumValue;

  @Override
  public void initialise() {
    charset = Charset.forName(defaultEncoding);
    streamingTypeEnumValue = StreamingType.valueOf(streamingType);
  }

  public Charset getCharset() {
    return charset;
  }

  public Integer getResponseTimeout() {
    return responseTimeout;
  }

  public TimeUnit getResponseTimeoutUnit() {
    return responseTimeoutUnit;
  }

  public StreamingType getStreamingType() {
    return streamingTypeEnumValue;
  }

  /**
   * Implementations can override this method to add different bindings parameters to the configuration. An example could be:
   * put("myConfigParam","myValueConfigParam");
   * 
   * @return the custom bindings added to the configuration or an empty map
   */
  public Map<String, Object> getBindings() {
    // Add custom bindings
    return emptyMap();
  }

  /**
   * @return the response interceptor descriptor added to the configuration or null
   */
  public HttpResponseInterceptorDescriptor getResponseInterceptorDescriptor() {
    return null;
  }

  public String getDefaultExpressionOutputMediaType() {
    // TODO RSDK-760: this method should be overridden in generated sub-classes
    return "application/json";
  }

}
