/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.model.builder.common;

import static com.mulesoft.connectivity.rest.commons.api.dw.DWBindings.BODY;
import static com.mulesoft.connectivity.rest.commons.api.dw.DWBindings.HEADERS;
import static com.mulesoft.connectivity.rest.commons.api.dw.DWBindings.PARAMETERS;
import static com.mulesoft.connectivity.rest.commons.api.dw.DWBindings.QUERIES;
import static com.mulesoft.connectivity.rest.commons.api.dw.DWBindings.URIS;
import static com.mulesoft.connectivity.rest.commons.internal.util.ResolverUtil.addGlobalDefinitionsToEvaluationContext;

import org.mule.runtime.api.metadata.TypedValue;

import com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext;
import com.mulesoft.connectivity.rest.commons.internal.model.common.SimpleEvaluationContext;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Evaluation context builder for an operation context.
 *
 * @since 1.0
 */
public class OperationEvaluationContextBuilder extends EvaluationContextBuilder {

  private final static String KEY_HEADERS = HEADERS.getBinding();
  private final static String KEY_URI_PARAMETERS = URIS.getBinding();
  private final static String KEY_QUERY_PARAMETERS = QUERIES.getBinding();
  private final static String KEY_AUXILIAR_PARAMETERS = PARAMETERS.getBinding();
  private final static String BODY_PARAMETERS = BODY.getBinding();

  private final Map<String, Object> headers = new HashMap<>();
  private final Map<String, Object> uriParameters = new HashMap<>();
  private final Map<String, Object> queryParameters = new HashMap<>();
  private final Map<String, Object> auxiliarParameters = new HashMap<>();
  private final Map<String, Object> body = new HashMap<>();

  public OperationEvaluationContextBuilder(EvaluationContext globalEvaluationContext) {
    super(globalEvaluationContext);
  }

  /**
   * Adds an URI parameter to the evaluation context being built. These get later accessed via
   * '{@link #KEY_URI_PARAMETERS}.{name}'
   */
  public OperationEvaluationContextBuilder uriParameter(String name, Object value) {
    setKeyValue(name, value, uriParameters);
    return this;
  }

  /**
   * Adds an query parameter to the evaluation context being built. These get later accessed via
   * '{@link #KEY_QUERY_PARAMETERS}.{name}'
   */
  public OperationEvaluationContextBuilder queryParameter(String name, Object value) {
    setKeyValue(name, value, queryParameters);
    return this;
  }

  /**
   * Adds an query parameter to the evaluation context being built. These get later accessed via
   * '{@link #KEY_AUXILIAR_PARAMETERS}.{name}'
   */
  public OperationEvaluationContextBuilder parameters(String name, Object value) {
    setKeyValue(name, value, auxiliarParameters);
    return this;
  }

  /**
   * Adds a header to the evaluation context being built. These get later accessed via '{@link #KEY_HEADERS}.{name}'
   */
  public OperationEvaluationContextBuilder header(String name, Object value) {
    setKeyValue(name, value, headers);
    return this;
  }

  public OperationEvaluationContextBuilder body(String name, Object value) {
    List<String> path = Arrays.asList(name.split("\\."));

    Map<String, Object> map = body;
    for (int i = 0; i < path.size() - 1; i++) {
      String key = path.get(i);
      if (!map.containsKey(key)) {
        setKeyValue(key, new HashMap<String, Object>(), map);
      }
      map = ((TypedValue<Map<String, Object>>) map.get(key)).getValue();
    }
    setKeyValue(path.get(path.size() - 1), value, map);

    return this;
  }

  public EvaluationContext build() {
    final EvaluationContext evaluationContext = new SimpleEvaluationContext(buildGlobalEvaluationContext().orElse(null));
    evaluationContext.define(KEY_HEADERS, headers);
    evaluationContext.define(KEY_URI_PARAMETERS, uriParameters);
    evaluationContext.define(KEY_QUERY_PARAMETERS, queryParameters);
    evaluationContext.define(BODY_PARAMETERS, body);
    evaluationContext.define(KEY_AUXILIAR_PARAMETERS, auxiliarParameters);
    addGlobalDefinitionsToEvaluationContext(evaluationContext);
    return evaluationContext;
  }
}
