/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.model.dataexpressions;

import static com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext.PROPERTY_CONFIGURATION;
import static com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext.PROPERTY_CONNECTION;
import static com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext.PROPERTY_EXPRESSION_LANGUAGE;
import static com.mulesoft.connectivity.rest.commons.internal.util.RequestStreamingUtils.doRequestAndConsumeString;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.http.api.HttpConstants;

import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext;
import com.mulesoft.connectivity.rest.commons.internal.model.common.Expression;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;

import java.util.Optional;

/**
 * An evaluable -executable- HTTP Request, with its corresponding path, method and bindings. It will use the provided context to
 * get the configuration and connection that will allow making the actual HTTP Request.
 *
 * @since 1.0
 */
public class HttpRequestDataExpression implements DataExpression {

  public static final String CUSTOM_BINDING = "customBinding";
  private final String path;
  private final String method;
  private final MediaType outputMediaType;
  private final HttpRequestDataExpressionBinding bindings;

  public HttpRequestDataExpression(String path, String method, MediaType outputMediaType,
                                   HttpRequestDataExpressionBinding bindings) {
    this.path = path;
    this.method = method;
    this.outputMediaType = outputMediaType;
    this.bindings = bindings;
  }

  @Override
  public Object evaluate(EvaluationContext evaluationContext) {

    RestConfiguration configuration =
        (RestConfiguration) evaluationContext.resolveProperty(PROPERTY_CONFIGURATION);
    RestConnection connection = (RestConnection) evaluationContext.resolveProperty(PROPERTY_CONNECTION);

    final RestRequestBuilder restRequestBuilder = getRestRequestBuilder(evaluationContext);

    return doRequestAndConsumeString(connection, configuration, restRequestBuilder, outputMediaType,
                                     (ExpressionLanguage) evaluationContext.resolveProperty(PROPERTY_EXPRESSION_LANGUAGE));
  }

  private String getPathTemplate() {
    return path;
  }

  private Optional<HttpRequestDataExpressionBinding> getBindings() {
    return Optional.ofNullable(bindings);
  }

  private RestRequestBuilder getRestRequestBuilder(EvaluationContext evaluationContext) {
    RestConfiguration configuration =
        (RestConfiguration) evaluationContext.resolveProperty(PROPERTY_CONFIGURATION);
    RestConnection connection = (RestConnection) evaluationContext.resolveProperty(PROPERTY_CONNECTION);
    final RestRequestBuilder restRequestBuilder =
        new RestRequestBuilder(connection.getBaseUri(), getPathTemplate(), HttpConstants.Method.valueOf(method.toUpperCase()));
    if (configuration.getResponseInterceptorDescriptor() != null) {
      restRequestBuilder.responseInterceptorDescriptor(configuration.getResponseInterceptorDescriptor());
    }
    addUriParams(evaluationContext, restRequestBuilder);

    getBindings().ifPresent(httpRequestDataExpressionBinding -> {
      httpRequestDataExpressionBinding.getHeaders().forEach(header -> restRequestBuilder
          .addHeader(header.getName(), getParameterValue(header.getValue(), evaluationContext)));
      httpRequestDataExpressionBinding.getQueryParameters().forEach(queryParameter -> restRequestBuilder
          .addQueryParam(queryParameter.getName(), getParameterValue(queryParameter.getValue(), evaluationContext)));
    });

    return restRequestBuilder;
  }

  private void addUriParams(EvaluationContext evaluationContext, RestRequestBuilder restRequestBuilder) {
    getBindings().ifPresent(httpRequestDataExpressionBinding -> httpRequestDataExpressionBinding.getUriParameters()
        .forEach(uriParameter -> restRequestBuilder.addUriParam(uriParameter.getName(),
                                                                getParameterValue(uriParameter.getValue(), evaluationContext))));
  }

  private String getParameterValue(Expression expression, EvaluationContext evaluationContext) {
    Object result = expression.evaluate(evaluationContext);

    // Using DW to obtaining the string value conversion from the expression+evaluationContext
    TypedValue<?> value = TypedValue.of(result);
    BindingContext build = BindingContext.builder()
        .addBinding(CUSTOM_BINDING, value)
        .build();
    ExpressionLanguage expressionLanguage = (ExpressionLanguage) evaluationContext.resolveProperty(PROPERTY_EXPRESSION_LANGUAGE);
    TypedValue<?> parameterValue = expressionLanguage.evaluate(CUSTOM_BINDING, DataType.STRING, build);

    return (String) parameterValue.getValue();
  }
}
