/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.model.builder.resolvers;

import com.mulesoft.connectivity.rest.commons.internal.model.resolvers.ResolverDeclaration;
import com.mulesoft.connectivity.rest.commons.internal.model.resolvers.ResolverDefinition;
import com.mulesoft.connectivity.rest.commons.internal.model.resolvers.ResolverParameter;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * Base class for a fluent resolver declaration builder.
 *
 * @param <T> The resolver definition builder that allows building the resolver definition for this declaration.
 * @param <U> The resolver definition the declaration will refer to.
 *
 * @since 1.0
 */
public abstract class ResolverDeclarationBuilder<T extends ResolverDefinitionBuilder<U>, U extends ResolverDefinition> {

  protected final String name;
  protected List<ResolverParameterBuilder> parameters;
  protected T definitionBuilder;

  public ResolverDeclarationBuilder(String name, T definitionBuilder) {
    if (name == null) {
      throw new IllegalArgumentException();
    }

    this.name = name;
    this.definitionBuilder = definitionBuilder;
    parameters = new ArrayList<>();
  }

  public ResolverDeclarationBuilder<T, U> definition(Consumer<T> resolverDefinitionBuilderConsumer) {
    resolverDefinitionBuilderConsumer.accept(definitionBuilder);
    return this;
  }

  public ResolverDeclaration<U> build() {
    return new ResolverDeclaration<>(getName(), buildParameters(), buildDefinition());
  }

  public ResolverDeclarationBuilder<T, U> parameter(String name,
                                                    Consumer<ResolverParameterBuilder> resolverParameterBuilderConsumer) {
    final ResolverParameterBuilder resolverParameterBuilder = new ResolverParameterBuilder(name);
    resolverParameterBuilderConsumer.accept(resolverParameterBuilder);
    parameters.add(resolverParameterBuilder);
    return this;
  }

  protected U buildDefinition() {
    if (definitionBuilder == null) {
      throw new IllegalArgumentException();
    }
    return definitionBuilder.build();
  }

  protected List<ResolverParameter> buildParameters() {
    return parameters.stream()
        .map(ResolverParameterBuilder::build)
        .collect(Collectors.toList());
  }

  protected String getName() {
    return name;
  }
}
