/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.model.common;

import static org.mule.runtime.api.el.BindingContext.builder;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.TypedValue;

import com.mulesoft.connectivity.rest.commons.internal.model.resolvers.ResolverDeclaration;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * The default implementation of an evaluation context.
 *
 * @since 1.0
 */
@SuppressWarnings("rawtypes")
public class SimpleEvaluationContext implements EvaluationContext {

  private final EvaluationContext globalEvaluationContext;

  private final Map<String, Object> properties;
  private final Map<String, Object> variables;
  private final Map<String, ResolverDeclaration> declarations;

  public SimpleEvaluationContext() {
    this(null);
  }

  public SimpleEvaluationContext(EvaluationContext globalEvaluationContext) {
    this.globalEvaluationContext = globalEvaluationContext;
    variables = new HashMap<>();
    properties = new HashMap<>();
    declarations = new HashMap<>();
  }

  @Override
  public Map<String, Object> getVariables() {
    return variables;
  }

  public Optional<EvaluationContext> getGlobalEvaluationContext() {
    return Optional.ofNullable(globalEvaluationContext);
  }

  @Override
  public Object resolve(String expression) {
    ExpressionLanguage expressionLanguage = (ExpressionLanguage) resolveProperty(PROPERTY_EXPRESSION_LANGUAGE);
    BindingContext.Builder bindingContextBuilder = builder();
    getVariables().forEach((s, o) -> bindingContextBuilder.addBinding(s, TypedValue.of(o)));

    final TypedValue<?> result = expressionLanguage.evaluate(expression, bindingContextBuilder.build());
    if (result == null) {
      throw new RuntimeException(String.format("Unresolvable expression '%s'", expression));
    } else {
      return result;
    }
  }

  @Override
  public Object resolveProperty(String property) {
    return Optional.ofNullable(properties.get(property))
        .orElseGet(() -> getGlobalEvaluationContext()
            .map(evaluationContext -> evaluationContext.resolveProperty(property)).orElse(null));
  }

  @Override
  public void defineProperty(String property, Object value) {
    properties.put(property, value);
  }

  @Override
  public void define(String variable, Object value) {
    variables.put(variable, value);
  }

  @Override
  public ResolverDeclaration resolveDeclaration(String declarationName) {
    final ResolverDeclaration result =
        Optional.ofNullable(declarations.get(declarationName))
            .orElseGet(() -> getGlobalEvaluationContext()
                .map(evaluationContext -> evaluationContext.resolveDeclaration(declarationName)).orElse(null));

    if (result == null) {
      throw new RuntimeException(String.format("Unresolvable declaration '%s'", declarationName));
    } else {
      return result;
    }
  }

  @Override
  public void declare(ResolverDeclaration resolverDeclaration) {
    declarations.put(resolverDeclaration.getName(), resolverDeclaration);
  }
}
