/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.util;

import static com.mulesoft.connectivity.rest.commons.api.configuration.StreamingType.ALWAYS;
import static com.mulesoft.connectivity.rest.commons.api.configuration.StreamingType.NEVER;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder.QueryParamFormat.MULTIMAP;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.stringValue;
import static java.lang.String.format;
import static java.lang.String.join;
import static org.mule.runtime.core.api.util.IOUtils.toByteArray;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.runtime.http.api.HttpHeaders.Names.TRANSFER_ENCODING;
import static org.mule.runtime.http.api.HttpHeaders.Values.CHUNKED;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.HttpConstants.Method;
import org.mule.runtime.http.api.domain.CaseInsensitiveMultiMap;
import org.mule.runtime.http.api.domain.entity.ByteArrayHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import com.mulesoft.connectivity.rest.commons.api.configuration.StreamingType;
import com.mulesoft.connectivity.rest.commons.api.operation.RequestParameters;
import com.mulesoft.connectivity.rest.commons.api.operation.queryparam.CommaQueryParamFormatter;
import com.mulesoft.connectivity.rest.commons.api.operation.queryparam.MultimapQueryParamFormatter;
import com.mulesoft.connectivity.rest.commons.api.operation.queryparam.QueryParamFormatter;

import java.io.InputStream;
import java.util.HashMap;
import java.util.List;

import org.slf4j.Logger;


/***
 * Builder of {@link HttpRequest}s. Handles ignoring null queryParams and Headers by default. Provides the ability of setting a
 * format for query parameter arrays. Handles request streaming mode (Content-Length and Transfer-Encoding headers).
 *
 * @since 1.0
 */
public class RestRequestBuilder {

  private static final Logger LOGGER = getLogger(RestRequestBuilder.class);

  private final String baseUri;
  private String path;
  private final Method method;
  private String fullUri;
  private final MultiMap<String, String> headers = new CaseInsensitiveMultiMap(false);
  private final MultiMap<String, String> queryParams = new MultiMap<>();
  private final HashMap<String, String> uriParams = new HashMap<>();
  private TypedValue<InputStream> body = null;
  private StreamingType streamingType = StreamingType.AUTO;
  private QueryParamFormat queryParamFormat = MULTIMAP;

  /**
   * Creates a preconfigured {@link RestRequestBuilder} per the given parameters.
   * <p>
   * This creates a new {@link RestRequestBuilder} and initializes its URI and method.
   * <p>
   * The returned builder can still to add/change additional information.
   *
   * @param baseUri the base uri to be used
   * @param path the path of the resource to request, relative to the base URI
   * @param method the request method
   */
  public RestRequestBuilder(String baseUri, String path, Method method) {
    this(baseUri, path, method, null);
  }

  /**
   * Creates a preconfigured {@link RestRequestBuilder} per the given parameters.
   * <p>
   * This creates a new {@link RestRequestBuilder} and initializes it's URI, method, custom query params and custom headers.
   * <p>
   * The returned builder can still to add/change additional information.
   *
   * @param baseUri the base uri to be used
   * @param path the path of the resource to request, relative to the base URI
   * @param method the request method
   * @param requestParameters the request parameters
   */
  public RestRequestBuilder(String baseUri, String path, Method method, RequestParameters requestParameters) {
    this.baseUri = baseUri;
    this.path = path;
    this.method = method;
    if (requestParameters != null) {
      this.headers.putAll(requestParameters.getCustomHeaders());
      this.queryParams.putAll(requestParameters.getCustomQueryParams());
    }
  }

  /***
   * Returns the full uri for this request. (Base Uri + Path)
   */
  public String getUri() {
    return buildRequestUri();
  }

  /***
   * Returns the HTTP verb to be used in this request
   */
  public String getMethod() {
    return method.name();
  }

  /***
   * Adds an uri parameter to the request builder. It will only be added if it's value is not null. URI parameters will be
   * replaced in the path where a placeholder matches its key. This method will use a best effort approach to convert the Object
   * to a valid String representation. If a specific format is needed, send the value as String.
   *
   * @param placeholderKey The header name.
   * @param value The header value. Will be converted to string.
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder addUriParam(String placeholderKey, Object value) {
    if (value != null) {
      uriParams.put(placeholderKey, stringValue(value));
    }
    return this;
  }

  /***
   * Adds multiple uri parameters (with the same name) to the request builder. All values are expected to be not null and will be
   * added using the same key. This method will use a best effort approach to convert the values to a valid String representation.
   * If a specific format is needed, send the values as String.
   *
   * @param key The header name.
   * @param values The header values.
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder addUriParams(String key, List<?> values) {
    if (values != null && !values.isEmpty()) {
      values.forEach(value -> addUriParam(key, value));
    }
    return this;
  }

  /***
   * Adds a header to the request builder. It will only be added if it's value is not null. This method will use a best effort
   * approach to convert the Object to a valid String representation. If a specific format is needed, send the value as String.
   *
   * @param key The header name.
   * @param value The header value. Will be converted to string.
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder addHeader(String key, Object value) {
    if (value != null) {
      headers.put(key, stringValue(value));
    }
    return this;
  }

  /***
   * Adds multiple headers (with the same name) to the request builder. All values are expected to be not null and will be added
   * using the same key. This method will use a best effort approach to convert the values to a valid String representation. If a
   * specific format is needed, send the values as String.
   *
   * @param key The header name.
   * @param values The header values.
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder addHeaders(String key, List<?> values) {
    if (values != null && !values.isEmpty()) {
      values.forEach(value -> addHeader(key, value));
    }
    return this;
  }

  /***
   * Adds all the headers present in a multimap to the request builder.
   * 
   * @param headers The headers to be added to the request builder.
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder headers(MultiMap<String, String> headers) {
    this.headers.putAll(headers);
    return this;
  }

  /***
   * Returns an immutable map containing the headers configured for this request.
   */
  public MultiMap<String, String> getHeaders() {
    return headers.toImmutableMultiMap();
  }

  /***
   * Adds a query parameter to the request builder. It will only be added if it's value is not null. This method will use a best
   * effort approach to convert the Object to a valid String representation. If a specific format is needed, send the value as
   * String.
   *
   * @param key The query parameter name.
   * @param value The query parameter value. Will be converted to string.
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder addQueryParam(String key, Object value) {
    if (value != null) {
      queryParams.put(key, stringValue(value));
    }
    return this;
  }

  /***
   * Adds multiple query parameters (with the same name) to the request builder. All values are expected to be not null and will
   * be added using the same key. This method will use a best effort approach to convert the values to a valid String
   * representation. If a specific format is needed, send the values as String.
   *
   * @param key The query parameter name.
   * @param values The query parameter value.
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder addQueryParams(String key, List<?> values) {
    if (values != null && !values.isEmpty()) {
      values.forEach(value -> addQueryParam(key, value));
    }
    return this;
  }

  /***
   * Adds all the query parameters present in a multimap to the request builder.
   * 
   * @param queryParams The query parameters to be added to the request builder.
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder queryParams(MultiMap<String, String> queryParams) {
    this.queryParams.putAll(queryParams);
    return this;
  }

  /***
   * Returns an immutable map containing the query parameters configured for this request.
   */
  public MultiMap<String, String> getQueryParams() {
    return queryParams.toImmutableMultiMap();
  }

  /***
   * COMMA: key=value1,value2,value3 MULTIMAP: key=value1&key=value2&key=value3
   */
  public enum QueryParamFormat {

    MULTIMAP(new MultimapQueryParamFormatter()), COMMA(new CommaQueryParamFormatter());

    private final QueryParamFormatter formatter;

    QueryParamFormat(QueryParamFormatter formatter) {
      this.formatter = formatter;
    }

    public QueryParamFormatter getFormatter() {
      return formatter;
    }
  }

  /***
   * Defines the format in which the query parameter arrays must be handled by this request.
   * 
   * @param queryParamFormat The query parameter array format
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder setQueryParamFormat(QueryParamFormat queryParamFormat) {
    this.queryParamFormat = queryParamFormat;
    return this;
  }


  /***
   * Defines the body and content streaming for this request.
   * 
   * @param body A {@link TypedValue} that contains the body to be sent in this request. Nullable.
   * @param streamingType The streaming type that must be used for this request's content.
   * @return The request builder so it can be used in a fluent way.
   */
  public RestRequestBuilder setBody(TypedValue<InputStream> body, StreamingType streamingType) {
    this.body = body;
    this.streamingType = streamingType;
    return this;
  }

  /**
   * Generates an {@link RestRequestBuilder} that reflects the request body. Sets the media-type and content-length headers for
   * this builder based on the entity and the request's streaming mode.
   *
   * @param body the request body (nullable)
   * @param streamingType the streaming mode for this request
   * @return an http entity to use in the request body. Null if the body must be empty.
   */
  private HttpEntity getStreamingConfiguredHttpEntity(TypedValue<InputStream> body, StreamingType streamingType) {
    if (body != null) {
      byte[] bytes = null;

      if (streamingType.equals(ALWAYS)) {
        headers.remove(CONTENT_LENGTH);
        headers.remove(TRANSFER_ENCODING);
        headers.put(TRANSFER_ENCODING, CHUNKED);
      } else if (streamingType.equals(NEVER)) {
        bytes = setNeverStreamingContentLength(body);
      } else {
        setAutoContentLengthHeader(body);
      }

      inferContentTypeFromBody(body);

      if (body.getValue() != null) {
        if (bytes != null) {
          return new ByteArrayHttpEntity(bytes);
        } else {
          return new InputStreamHttpEntity(body.getValue());
        }
      }
    }
    return null;
  }

  private void inferContentTypeFromBody(TypedValue<InputStream> body) {
    if (!headers.containsKey(CONTENT_TYPE)) {
      MediaType mediaType = body.getDataType().getMediaType();
      if (mediaType != null && !mediaType.getPrimaryType().equals("*")) {
        headers.put(CONTENT_TYPE, mediaType.toRfcString());
      }
    }
  }

  /**
   * Sets the content length header value for this requests, reading the body to get it if necessary. If the body has been read
   * this method returns it's bytes.
   * 
   * @param body The request body
   * @return A byte[] containing the body if the input stream has been read.
   */
  private byte[] setNeverStreamingContentLength(TypedValue<InputStream> body) {
    String customLength = headers.get(CONTENT_LENGTH);

    byte[] bytes = null;

    if (customLength == null) {
      if (body.getByteLength().isPresent()) {
        addHeader(CONTENT_LENGTH, body.getByteLength().getAsLong());
      } else if (body.getValue() != null) {
        bytes = toByteArray(body.getValue());
        addHeader(CONTENT_LENGTH, bytes.length);
      }
    }

    headers.remove(TRANSFER_ENCODING);

    return bytes;
  }

  private void setAutoContentLengthHeader(TypedValue<InputStream> body) {
    String customLength = headers.get(CONTENT_LENGTH);
    boolean isChunked = CHUNKED.equals(headers.get(TRANSFER_ENCODING));

    if (body.getByteLength().isPresent()) {
      boolean addHeader = true;
      String length = stringValue(body.getByteLength().getAsLong());
      if (customLength != null) {
        LOGGER.warn("Invoking URI {} with body of known length {}. However, a {} header with value {} was manually specified. "
            + "Will proceed with the custom value.",
                    getUri(), length, CONTENT_LENGTH, customLength);

        addHeader = false;
      }

      if (isChunked) {
        LOGGER.debug("Invoking URI {} with a manually set {}: {} header, even though body is of known length {}. "
            + "Skipping automatic addition of {} header", getUri(), TRANSFER_ENCODING, CHUNKED, length, CONTENT_LENGTH);
        addHeader = false;
      }

      if (addHeader) {
        headers.put(CONTENT_LENGTH, length);
      }

    } else if (customLength == null && !isChunked) {
      headers.put(TRANSFER_ENCODING, CHUNKED);
    }
  }

  /***
   * @return The configured HttpRequest
   */
  public HttpRequest build() {
    HttpEntity httpEntity = getStreamingConfiguredHttpEntity(body, streamingType);

    HttpRequestBuilder builder = HttpRequest.builder(true)
        .uri(getUri())
        .method(method)
        .queryParams(queryParamFormat.getFormatter().format(queryParams))
        .headers(headers);

    if (httpEntity != null) {
      builder.entity(httpEntity);
    }

    return builder.build();
  }

  /**
   * Sets the path section for the request URI
   *
   * @param path the path of the request.
   */
  public void setPath(String path) {
    this.path = path;
  }

  /**
   * Sets a full request URI that overrides the configured baseUri and path.
   * 
   * @param fullUri to be set as the request URI
   */
  public void setFullUri(String fullUri) {
    this.fullUri = fullUri;
  }

  private String buildRequestUri() {
    if (isNotBlank(fullUri)) {
      return fullUri;
    }

    String localPath = path;
    String localBaseUri = baseUri;

    boolean pathStartsWithSlash = localPath != null && localPath.startsWith("/");
    boolean baseEndsInSlash = localBaseUri.endsWith("/");

    if (pathStartsWithSlash && baseEndsInSlash) {
      localPath = localPath.substring(1);
    } else if (!pathStartsWithSlash && !baseEndsInSlash) {
      localBaseUri += '/';
    }

    localPath = applyUriParameters(localPath);

    return localBaseUri + localPath;
  }

  private String applyUriParameters(String path) {
    for (String key : uriParams.keySet()) {
      path = path.replace(format("{%s}", key), uriParams.get(key));
    }
    return path;
  }
}
