/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.interception.expression;

import static java.util.Objects.requireNonNull;

import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.interception.HttpResponseInterceptor;
import com.mulesoft.connectivity.rest.commons.api.interception.InterceptionHttpRequest;

/**
 * A {@link HttpResponseInterceptor} implementation that applies a precondition and if satisfied and the delegate matches too it
 * delegates the execution to a nested {@link HttpResponseInterceptor}.
 * <p/>
 * This is useful when grouping {@link HttpResponseInterceptor}s by a precondition like: statusCode == 200
 *
 * @since 1.0
 */
public class PreconditionHttpResponseInterceptorDelegate implements HttpResponseInterceptor {

  private Matcher precondition;
  private HttpResponseInterceptor delegate;

  public PreconditionHttpResponseInterceptorDelegate(Matcher precondition,
                                                     HttpResponseInterceptor delegate) {
    requireNonNull(precondition);
    requireNonNull(delegate);

    this.precondition = precondition;
    this.delegate = delegate;
  }

  @Override
  public boolean match(InterceptionHttpRequest httpRequest, HttpResponse httpResponse, ExpressionLanguage expressionLanguage) {
    return precondition.match(httpRequest, httpResponse) && delegate.match(httpRequest, httpResponse, expressionLanguage);
  }

  /**
   * Defines a predicate condition.
   */
  public interface Matcher {

    boolean match(InterceptionHttpRequest httpRequest, HttpResponse httpResponse);
  }

  @Override
  public HttpResponse intercept(InterceptionHttpRequest httpRequest, HttpResponse httpResponse,
                                ExpressionLanguage expressionLanguage) {
    return delegate.intercept(httpRequest, httpResponse, expressionLanguage);
  }

}
