/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isBlank;
import static org.mule.runtime.api.metadata.DataType.STRING;

import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;

import java.util.List;
import java.util.Map;
import java.util.function.Function;

/**
 * Implementation of {@link RestPagingProvider} for APIs which do paging based on a marker provided in the response of each
 * request, acting as a marker or position inside the stream of results.
 * <p>
 * Request number 1 will obtain the first page plus a marker embedded in the response. Said marker will be added as a query param
 * in request number 2, which will reply with the next page and a new marker which overwrites the prior one. The process continues
 * until no new marker is provided or the next page comes back empty.
 *
 * @since 1.0
 */
public class MarkerPagingProvider extends RestPagingProvider {

  private final String markerParamName;
  private final String nextMarkerExpression;

  private boolean firstPage = true;
  private String nextMarker;

  /**
   * Creates a new instance
   *
   * @param markerParamName the name of the query parameter in which the marker is placed in the request
   * @param nextMarkerExpression the DW expression used to extract the next marker from the response
   * @param requestFactory a {@link Function} to generate the request to be used on each page request. Each invocation should
   *        yield a different instance
   * @param expressionLanguage the app's {@link ExpressionLanguage}
   * @param payloadExpression a DW expression to extract the data from the response
   * @param defaultMediaType the {@link MediaType} for the page items if the server doesn't specify one
   * @param responseTimeout the timeout for each request
   */
  public MarkerPagingProvider(String markerParamName,
                              String nextMarkerExpression,
                              Function<RestConnection, RestRequestBuilder> requestFactory,
                              ExpressionLanguage expressionLanguage,
                              String payloadExpression,
                              MediaType defaultMediaType,
                              Map<String, Object> parameterBindings,
                              Map<String, Object> customParameterBindings,
                              int responseTimeout) {
    super(requestFactory, expressionLanguage, payloadExpression,
          defaultMediaType, parameterBindings, customParameterBindings, responseTimeout);
    this.markerParamName = markerParamName;
    this.nextMarkerExpression = nextMarkerExpression;
  }

  @Override
  protected void configureRequest(RestRequestBuilder requestBuilder) {
    if (!firstPage) {
      if (isBlank(nextMarker)) {
        stopPaging();
        return;
      }

      requestBuilder.addQueryParam(markerParamName, nextMarker);
    }
  }

  @Override
  protected void onPage(List<TypedValue<String>> page, TypedValue<String> rawPage, HttpResponseAttributes responseAttributes) {
    firstPage = false;

    if (isBlank(rawPage.getValue())) {
      nextMarker = null;
      stopPaging();
    } else {
      nextMarker = (String) evaluate(rawPage, nextMarkerExpression, STRING, responseAttributes).getValue();
    }
  }
}
