/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.interception.expression;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.resolveCharset;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.DataType.OBJECT;
import static org.mule.runtime.api.metadata.DataType.fromType;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.server.HttpServerProperties.PRESERVE_HEADER_CASE;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.CompiledExpression;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.el.ExpressionLanguageSession;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.interception.HttpRequest;
import com.mulesoft.connectivity.rest.commons.api.interception.HttpResponseInterceptor;
import com.mulesoft.connectivity.rest.commons.api.streaming.StreamingHelper;
import com.mulesoft.connectivity.rest.commons.internal.interception.model.HttpEntityCursorStreamProviderBased;
import com.mulesoft.connectivity.rest.commons.internal.interception.model.RepeatableHttpResponse;
import com.mulesoft.connectivity.rest.commons.internal.util.FromCursorProviderInputStream;

import java.util.Optional;
import java.util.OptionalLong;

import org.slf4j.Logger;

/**
 * A {@link HttpResponseInterceptor} implementation based on DW expressions. It allows use of DW expressions to define the
 * matching condition and transformations for statusCode, reasonPhrase, headers and body.
 * <p/>
 * {@link BindingContext} provided on each expression has the original {@link HttpResponse} data with the following variables:
 * <ul>
 * <li>statusCode: int</li>
 * <li>reasonPhrase: String</li>
 * <li>headers: MultiMap<String, String></li>
 * <li>body: InputStream</li>
 * </ul>
 */
public class ExpressionHttpResponseInterceptor extends BaseExpressionHttpResponseInterceptor {

  private static final Logger LOGGER = getLogger(ExpressionHttpResponseInterceptor.class);

  private String matchExpression;

  private String statusCodeExpression;
  private String reasonPhraseExpression;
  private String headersExpression;
  private String bodyExpression;

  private ExpressionHttpResponseInterceptor(String matchExpression,
                                            String statusCodeExpression,
                                            String reasonPhraseExpression,
                                            String headersExpression,
                                            String bodyExpression,
                                            MediaType defaultResponseMediaType,
                                            ExpressionLanguage expressionLanguage,
                                            StreamingHelper streamingHelper) {
    super(defaultResponseMediaType, expressionLanguage, streamingHelper);
    requireNonNull(matchExpression);

    this.matchExpression = matchExpression;

    this.statusCodeExpression = statusCodeExpression;
    this.reasonPhraseExpression = reasonPhraseExpression;
    this.headersExpression = headersExpression;
    this.bodyExpression = bodyExpression;
  }

  private boolean match(ExpressionLanguageSession session) {
    return resolveBoolean(evaluate(matchExpression,
                                   session,
                                   OBJECT).getValue(),
                          matchExpression);
  }

  private static boolean resolveBoolean(Object result, String expression) {
    if (result == null) {
      return false;
    } else {
      Object value = result;
      if (value instanceof Boolean) {
        return (Boolean) value;
      } else if (value instanceof String) {
        if (value.toString().equalsIgnoreCase("false")) {
          return false;
        } else if (result.toString().equalsIgnoreCase("true")) {
          return true;
        } else {
          LOGGER.warn("Expression: '" + expression + "', returned an non-boolean result: '" + result + "'. Returning: false");
          return false;
        }
      } else {
        LOGGER.warn("Expression: " + expression + ", returned an non-boolean result: '" + result + "'. Returning: false");
        return false;
      }
    }
  }

  @Override
  protected RepeatableHttpResponse doIntercept(HttpRequest httpRequest, RepeatableHttpResponse repeatableHttpResponse,
                                               ExpressionLanguageSession session, BindingContext bindingContext,
                                               ExpressionLanguage expressionLanguage) {
    if (match(session)) {
      int statusCode = (int) getStatusCodeExpression()
          .map(expression -> evaluate(expression, session,
                                      fromType(Integer.class))
                                          .getValue())
          .orElseGet(() -> repeatableHttpResponse.getStatusCode());
      String reasonPhrase = (String) getReasonPhraseExpression()
          .map(expression -> evaluate(expression, session,
                                      fromType(String.class))
                                          .getValue())
          .orElseGet(() -> repeatableHttpResponse.getReasonPhrase());

      MultiMap<String, String> headers =
          (MultiMap<String, String>) getHeadersExpression()
              .map(expression -> evaluate(headersExpression, session,
                                          fromType(MultiMap.class)).getValue())
              .orElseGet(() -> repeatableHttpResponse.getHeaders());

      HttpEntity entity;
      if (getBodyExpression().isPresent()) {
        String bodyExpression = getBodyExpression().get();
        TypedValue<Object> transformedBody = evaluateBodyExpression(bodyExpression, repeatableHttpResponse,
                                                                    session, bindingContext,
                                                                    expressionLanguage);

        if (transformedBody.getValue() instanceof CursorStreamProvider) {
          // If the expression generates an output it will be a CursorStreamProvider
          CursorStreamProvider cursorStreamProvider = (CursorStreamProvider) transformedBody.getValue();
          // Only if it is not the same CSP that was set to the "body" binding context.
          // This means DW has created a CSP as a transformation has been made and the body changed.
          if (cursorStreamProvider != repeatableHttpResponse.getRepeatableEntity().getCursorStreamProvider()) {
            // Just wrap the cursorStreamProvider with a HttpEntityCursorStreamProviderBased therefore the repeatableHttpResponse
            // gets used in the RepeatableHttpResponse.
            entity = new HttpEntityCursorStreamProviderBased(cursorStreamProvider,
                                                             repeatableHttpResponse.getEntity().getBytesLength().isPresent()
                                                                 ? transformedBody.getByteLength()
                                                                 : OptionalLong.empty());
            headers = updateContentLengthHeader(headers, transformedBody.getByteLength());

            // Close original repeatable response which will close and release the CSP.
            repeatableHttpResponse.close();
          } else {
            // Just use the same HttpEntity from the repeatable response.
            entity = repeatableHttpResponse.getRepeatableEntity();
          }
        } else {
          throw new MuleRuntimeException(
                                         createStaticMessage(
                                                             "HTTP response interceptor expression for body has generated a non recognizable output: '%s'. This is probably a bug.",
                                                             transformedBody.getValue().getClass().getName()));
        }
      } else {
        // The body expression transformation is not defined, so we close the RepeatableHttpResponse and gets its HttpEntity as
        // body transformation expression is not defined.
        entity = repeatableHttpResponse.getEntity();
      }
      return RepeatableHttpResponse.newRepeatableHttpResponse(HttpResponse.builder()
          .statusCode(statusCode)
          .reasonPhrase(reasonPhrase)
          .entity(entity)
          .headers(headers)
          .build(), streamingHelper);
    } else {
      return repeatableHttpResponse;
    }
  }

  private TypedValue<Object> evaluateBodyExpression(String bodyExpression, HttpResponse httpResponse,
                                                    ExpressionLanguageSession session,
                                                    BindingContext bindingContext,
                                                    ExpressionLanguage expressionLanguage) {
    CompiledExpression compile = expressionLanguage.compile(bodyExpression, bindingContext);
    MediaType mediaType = compile.outputType().orElse(getMediaType(httpResponse));
    return (TypedValue<Object>) session
        .evaluate(bodyExpression, DataType.builder()
            .mediaType(mediaType)
            .charset(resolveCharset(empty(), mediaType))
            .build());
  }

  private MultiMap<String, String> updateContentLengthHeader(MultiMap<String, String> headers, OptionalLong length) {
    if (!headers.containsKey(CONTENT_LENGTH) && !headers.containsKey(CONTENT_LENGTH.toLowerCase())) {
      return headers;
    }
    MultiMap.StringMultiMap modifiedHeaders = new MultiMap.StringMultiMap();
    for (String key : headers.keySet()) {
      if (!key.equalsIgnoreCase(CONTENT_LENGTH)) {
        modifiedHeaders.put(key, headers.getAll(key));
      }
    }

    modifiedHeaders.put(PRESERVE_HEADER_CASE ? CONTENT_LENGTH : CONTENT_LENGTH.toLowerCase(),
                        String.valueOf(length.getAsLong()));
    return modifiedHeaders;
  }

  private Optional<String> getBodyExpression() {
    return ofNullable(bodyExpression);
  }

  private Optional<String> getHeadersExpression() {
    return ofNullable(headersExpression);
  }

  private Optional<String> getStatusCodeExpression() {
    return ofNullable(statusCodeExpression);
  }

  private Optional<String> getReasonPhraseExpression() {
    return ofNullable(reasonPhraseExpression);
  }

  /**
   * @return a {@link ExpressionHttpResponseInterceptorBuilder} to create a {@link ExpressionHttpResponseInterceptor}.
   */
  public static ExpressionHttpResponseInterceptorBuilder builder() {
    return new ExpressionHttpResponseInterceptorBuilder();
  }

  /**
   * Builder for a {@link ExpressionHttpResponseInterceptor}.
   */
  public static class ExpressionHttpResponseInterceptorBuilder {

    private String matchExpression;

    protected String statusCodeExpression;
    protected String reasonPhraseExpression;
    protected String headersExpression;
    protected String bodyExpression;

    protected MediaType defaultResponseMediaType;

    protected ExpressionLanguage expressionLanguage;
    private StreamingHelper streamingHelper;

    public ExpressionHttpResponseInterceptorBuilder matchExpression(String matchExpression) {
      requireNonNull(matchExpression);
      this.matchExpression = matchExpression;
      return this;
    }

    /**
     * Sets the expression for resolving the status code.
     *
     * @param statusCodeExpression a DW expression that resolves the status code.
     * @return this
     */
    public ExpressionHttpResponseInterceptorBuilder statusCodeExpression(String statusCodeExpression) {
      requireNonNull(statusCodeExpression);
      this.statusCodeExpression = statusCodeExpression;
      return this;
    }

    /**
     * Sets the expression for resolving the reason phrase.
     *
     * @param reasonPhraseExpression a DW expression that resolves the reason phrase.
     * @return this
     */
    public ExpressionHttpResponseInterceptorBuilder reasonPhraseExpression(String reasonPhraseExpression) {
      requireNonNull(reasonPhraseExpression);
      this.reasonPhraseExpression = reasonPhraseExpression;
      return this;
    }

    public ExpressionHttpResponseInterceptorBuilder headersExpression(String headersExpression) {
      requireNonNull(headersExpression);
      this.headersExpression = headersExpression;
      return this;
    }

    /**
     * Sets the expression for resolving the body.
     *
     * @param bodyExpression a DW expression that resolves the body.
     * @return this
     */
    public ExpressionHttpResponseInterceptorBuilder bodyExpression(String bodyExpression) {
      requireNonNull(bodyExpression);
      this.bodyExpression = bodyExpression;
      return this;
    }

    /**
     * Defines the default response media type for the {@link HttpResponse} that is intercepted by the interceptor defined by this
     * builder.
     *
     * @param defaultResponseMediaType default response media type for the {@link HttpResponse} that is intercepted by the
     *        interceptor defined by this builder.
     * @return this
     */
    public ExpressionHttpResponseInterceptorBuilder defaultResponseMediaType(MediaType defaultResponseMediaType) {
      requireNonNull(defaultResponseMediaType);
      this.defaultResponseMediaType = defaultResponseMediaType;
      return this;
    }


    /**
     * Sets the Mule {@link ExpressionLanguage} to resolve the expressions.
     *
     * @param expressionLanguage {@link ExpressionLanguage} to resolve the expressions.
     * @return this
     */
    public ExpressionHttpResponseInterceptorBuilder expressionLanguage(ExpressionLanguage expressionLanguage) {
      requireNonNull(expressionLanguage);
      this.expressionLanguage = expressionLanguage;
      return this;
    }

    public ExpressionHttpResponseInterceptorBuilder streamingHelper(StreamingHelper streamingHelper) {
      requireNonNull(streamingHelper);
      this.streamingHelper = streamingHelper;
      return this;
    }

    /**
     * @return a instance of {@link ExpressionHttpResponseInterceptor}.
     */
    public ExpressionHttpResponseInterceptor build() {
      return new ExpressionHttpResponseInterceptor(matchExpression, statusCodeExpression, reasonPhraseExpression,
                                                   headersExpression,
                                                   bodyExpression, defaultResponseMediaType, expressionLanguage, streamingHelper);
    }

  }

}
