/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.backoff;

import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.function.Supplier;

public abstract class RandomizedExponentialBackoffCaller<T> {

  private static final Logger LOGGER = LoggerFactory.getLogger(RandomizedExponentialBackoffCaller.class);

  private final RetriableCallerConfiguration configuration;

  public RandomizedExponentialBackoffCaller(RetriableCallerConfiguration configuration) {
    this.configuration = configuration;
  }

  public T call(Supplier<Result<T>> resultSupplier) throws InterruptedException {
    String retriableRequestId = Long.toString(System.nanoTime());
    Result<T> result = resultSupplier.get();

    int retries = 0;

    while (isRetriable(result) && configuration.getMaxRetries() > 0 && retries < configuration.getMaxRetries()) {
      retries++;
      LOGGER.info("request will be retried. retries: {} maxRetries: {} retriableRequestId: {} reason: {} ", retries,
                  configuration.getMaxRetries(), retriableRequestId, ((Exception) result.getResult()).getMessage());
      sleep(retries, retriableRequestId);

      result = resultSupplier.get();
    }

    if (result.isError()) {
      if (retries == configuration.getMaxRetries())
        LOGGER.error("retries for this call were exhausted. maxRetries: {} retriableRequestId: {}", configuration.getMaxRetries(),
                     retriableRequestId);

      LOGGER.error("response is error. retriableRequestId: {} reason: {}", retriableRequestId,
                   ((Exception) result.getResult()).getMessage());
      throw (RuntimeException) result.getResult();
    }

    LOGGER.info("response success. retries: {} maxRetries: {} retriableRequestId: {}", retries, configuration.getMaxRetries(),
                retriableRequestId);
    return result.getResult();
  }

  private void sleep(int retries, String retriableRequestId) throws InterruptedException {
    long randomizedExponentialBackoff = getRandomizedExponentialWaitingTime(retries);
    long delay = getSleepWaitingTime(configuration.getMaxWaitingValue(), randomizedExponentialBackoff);
    try {
      LOGGER.info("sleeping before retry: retryNumber {} randomlyExponentialDelay {} finalDelay {} retriableRequestId: {}",
                  retries, randomizedExponentialBackoff, delay, retriableRequestId);
      Thread.sleep(delay);
    } catch (InterruptedException e) {
      throw new InterruptedException(e.getMessage());
    }
  }

  @VisibleForTesting
  protected long getSleepWaitingTime(long maxWaitingValue, long randomizedExponentialBackoff) {
    return Math.min(maxWaitingValue, randomizedExponentialBackoff);
  }

  @VisibleForTesting
  protected long getRandomizedExponentialWaitingTime(int retries) {
    double exponential = Math.pow(2, retries);
    return (long) (Math.floor(Math.random() * exponential) * 1000);
  }

  public abstract boolean isRetriable(Result<T> result);

}
